"""
Two-Stage Promising Review Mode Implementation

Simplified version that inherits from TwoStageReviewConference.
The only difference is selecting papers with HIGHEST scores instead of closest to 0.
"""

import sys
import os
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import numpy as np
from review_modes.two_stage_review import TwoStageReviewConference

class PromisingReviewConference(TwoStageReviewConference):
    """
    Two-Stage Promising Review Conference System

    Inherits all functionality from TwoStageReviewConference.
    The key difference: Stage 2 selects papers with HIGHEST scores (most promising)
    instead of closest to 0 (most ambiguous).
    """

    def select_stage2_papers(self):
        """
        Select papers for stage 2 based on stage 1 scores (HIGHEST scores = most promising)

        KEY DIFFERENCE: This selects papers with highest scores instead of closest to 0
        """
        print(f"\n=== Selecting Papers for Stage 2 (Promising Strategy) ===")

        if self.stage1_decisions is None:
            print("Error: Stage 1 decisions not available")
            return

        # Sort papers by score (DESCENDING = highest scores first)
        # This is the key difference from TwoStageReviewConference
        paper_indices = np.argsort(self.stage1_decisions)[::-1]  # [::-1] for descending order

        # Select top eta proportion of highest scoring papers
        num_stage2_papers = min(self.stage2_paper_num, len(paper_indices))
        stage2_paper_indices = paper_indices[:num_stage2_papers]

        self.stage2_papers = [self.papers[i] for i in stage2_paper_indices]
        self.stage2_paper_indices = stage2_paper_indices  # Store for later use

        print(f"Selected {len(self.stage2_papers)} highest-scoring (most promising) papers for stage 2")
        sample_scores = [f"{self.stage1_decisions[i]:.3f}" for i in stage2_paper_indices[:5]]
        print(f"Stage 2 paper scores (stage 1): {sample_scores}...")

    def run_promising_review(self):
        """
        Run complete two-stage promising review process

        Same as parent class run_two_stage_review, just renamed for clarity
        """
        print("=== Starting Two-Stage Promising Review Mode ===")

        # 1. Generate papers and reviewers
        self.generate_papers_and_reviewers()

        # 2. Stage 1: Assign and conduct reviews
        self.assign_stage1_reviews()
        self.conduct_stage1_reviews()
        self.apply_stage1_bp()

        # 3. Select papers for stage 2 (PROMISING strategy)
        self.select_stage2_papers()

        # 4. Stage 2: Assign and conduct reviews (if there are papers)
        if len(self.stage2_papers) > 0:
            self.assign_stage2_reviews()
            self.conduct_stage2_reviews()

        # 5. Final BP on combined results
        self.apply_final_bp()

        # 6. Evaluate performance
        results = self.evaluate_performance()

        print("=== Two-Stage Promising Review Mode Completed ===")
        return results


def main():
    """
    Test two-stage promising review mode
    """
    # Test configurations
    test_configs = [
        {
            'paper_num': 30,
            'reviewer_num': 20,
            'reviewer_quality': 0.8,
            'eta': 0.3,  # 30% of papers go to stage 2
            'lambda_per_paper': 5,
            'stage1_reviewer_ratio': 0.6,  # 60% reviewers in stage 1 (now irrelevant)
            'lambda1': 3  # NEW: explicit lambda1
        },
        {
            'paper_num': 40,
            'reviewer_num': 25,
            'reviewer_quality': 0.9,
            'eta': 0.4,  # 40% of papers go to stage 2
            'lambda_per_paper': 6,
            'stage1_reviewer_ratio': 0.5,  # 50% reviewers in stage 1 (now irrelevant)
            'lambda1': 4  # NEW: explicit lambda1
        }
    ]

    print("Starting Two-Stage Promising Review Mode Testing\n")

    for i, config in enumerate(test_configs):
        print(f"\n{'='*60}")
        print(f"Test Configuration {i+1}: {config}")
        print(f"{'='*60}")

        # Create and run two-stage promising review system
        conference = PromisingReviewConference(**config)
        results = conference.run_promising_review()

        # Output summary
        if results:
            print(f"\n[Config {i+1} Summary] Accuracy: {results['accuracy']:.3f}, "
                  f"F1: {results['f1_score']:.3f}, Total Reviews: {results['total_reviews']}")


if __name__ == "__main__":
    main()