"""
Test script for verifying new lambda parameter setting in all three review modes
"""

from direct_review import DirectReviewConference
from two_stage_review import TwoStageReviewConference
from promising_review import PromisingReviewConference
import numpy as np

def test_direct_review():
    """Test direct review mode with different lambda values"""
    print("\n" + "="*60)
    print("Testing Direct Review Mode with Lambda Settings")
    print("="*60)

    test_lambdas = [3, 6, 9, 12, 15]

    for lambda_val in test_lambdas:
        print(f"\n--- Testing with λ = {lambda_val} ---")
        conf = DirectReviewConference(
            paper_num=20,
            reviewer_num=15,
            reviewer_quality=0.8,
            lambda_per_paper=lambda_val
        )

        print(f"Lambda per paper: {conf.lambda_per_paper}")
        print(f"Reviews per paper: {conf.reviews_per_paper}")
        print(f"Avg tasks per reviewer: {conf.avg_tasks_per_reviewer:.1f}")

        # Run the review process
        conf.generate_papers_and_reviewers()
        conf.assign_reviews_direct()

        # Verify the assignment
        paper_review_counts = [len(paper.reviews) for paper in conf.papers]
        reviewer_loads = [len(reviewer.reviews) for reviewer in conf.reviewers]

        print(f"Actual reviews per paper: min={min(paper_review_counts)}, max={max(paper_review_counts)}")
        print(f"Actual reviewer workloads: min={min(reviewer_loads)}, max={max(reviewer_loads)}")

def test_two_stage_ambiguous():
    """Test two-stage ambiguous review mode with different lambda values"""
    print("\n" + "="*60)
    print("Testing Two-Stage Ambiguous Review Mode with Lambda Settings")
    print("="*60)

    test_lambdas = [3, 6, 9, 12, 15]

    for lambda_val in test_lambdas:
        print(f"\n--- Testing with λ = {lambda_val} ---")
        conf = TwoStageReviewConference(
            paper_num=30,
            reviewer_num=20,
            reviewer_quality=0.8,
            eta=0.5,
            lambda_per_paper=lambda_val,
            stage1_reviewer_ratio=0.6
        )

        print(f"Lambda per paper: {conf.lambda_per_paper}")
        print(f"Lambda1 (auto): {conf.lambda1}")
        print(f"Lambda2 (calculated): {conf.lambda2:.1f}")
        print(f"Constraint check: λ1 + η*λ2 = {conf.lambda1 + conf.eta * conf.lambda2:.1f} (should equal {lambda_val})")

def test_two_stage_promising():
    """Test two-stage promising review mode with different lambda values"""
    print("\n" + "="*60)
    print("Testing Two-Stage Promising Review Mode with Lambda Settings")
    print("="*60)

    test_lambdas = [3, 6, 9, 12, 15]

    for lambda_val in test_lambdas:
        print(f"\n--- Testing with λ = {lambda_val} ---")
        conf = PromisingReviewConference(
            paper_num=30,
            reviewer_num=20,
            reviewer_quality=0.8,
            eta=0.5,
            lambda_per_paper=lambda_val,
            stage1_reviewer_ratio=0.6
        )

        print(f"Lambda per paper: {conf.lambda_per_paper}")
        print(f"Lambda1 (auto): {conf.lambda1}")
        print(f"Lambda2 (calculated): {conf.lambda2:.1f}")
        print(f"Constraint check: λ1 + η*λ2 = {conf.lambda1 + conf.eta * conf.lambda2:.1f} (should equal {lambda_val})")

def main():
    """Main test function"""
    print("Testing New Lambda Parameter Settings in All Review Modes")
    print("This test verifies that lambda can be properly set during initialization")

    # Test all three modes
    test_direct_review()
    test_two_stage_ambiguous()
    test_two_stage_promising()

    print("\n" + "="*60)
    print("All Lambda Parameter Tests Completed")
    print("="*60)

if __name__ == "__main__":
    main()