# Code for Loss Adapted Plasticity

This directory contains the code required to run the experiments to produce the results presented in the paper "Loss Adapted Plasticity: Learning From Data With Unreliable Sources"

To get started and download all dependencies, run:

    pip3 install -r requirements.txt 

## LAP Optimisers

The key code in this directory relates to the LAP version of Adam and SGD (written in pytorch), which can be used in replacement with previous uses of Adam and SGD, with the only difference being that they need to be passed the loss and the source when performing the step.

In current code, the optimisation process looks as follows:

```python
inputs, labels = data_batch
# ======= forward ======= 
outputs = model(inputs)
loss = criterion(outputs, labels)
# ======= backward =======
loss.backward()
optimizer.step()
```

When using a LAP optimiser, this needs to be changed to (with the key difference being the optimiser step ```optimizer.step(loss, source)```):

```python
inputs, labels = data_batch
# ======= forward ======= 
outputs = model(inputs)
loss = criterion(outputs, labels)
# ======= backward =======
loss.backward()
optimizer.step(loss, source)
```

The code for these optimizers can be found in ```./my_code/training_utils/Optimizers.py```, as classes AdamLAP and SGDLAP. They can be imported using the code:

```python
from my_code.training_utils.Optimizers import AdamLAP, SGDLAP
```

A better name for the package and a more easily available implementation of AdamLAP and SGDLAP (made available on GitHub) will be available upon completion of the review process.

## Training

To train the models, please use the ```hparam.py``` file, with the correct arguments for that given experiment. 


The corruption and their corresponding short name is given in the list:

- No Corruption : ```no_c```
- Chunk Shuffle : ```c_cs```
- Random Label : ```c_rl```
- Batch Label Shuffle : ```c_lbs```
- Batch Label Flip : ```c_lbf```
- Added Noise : ```c_ns```
- Replace With Noise : ```c_no```


The optional arguments when training using LAP are:

- Depression Strength: ```--depression-strength```
    - This is the strength of the depression applied and is defined as $d_\zeta$ in the paper.
- Strictness Parameter: ```--strictness```
    - This is the strictness parameter defined as $\lambda$ in the paper.
- Loss History Length:  ```--lap-n```
    - This is the number of losses to store for each source when making the depression calculations.
- Hold Off Value: ```--hold-off```
    - This is the number of calls to the depression function before depression will be applied. It allows you to control when depression is applied.



### Training on CIFAR-10

For training the LAP trained model, please run:

    python hparam.py --model-name Conv3Net-[Corruption Short Name]-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name cifar10 --n-sources 10 --n-corrupt-sources 4 --source-size 128 --depression-strength 1.0 --strictness 0.8 --lap-n 25 --n-epochs 25 -v

For standard model training, please use (the key difference being ```--depression-strength 0.0```):

    python hparam.py --model-name Conv3Net-[Corruption Short Name]-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name cifar10 --n-sources 10 --n-corrupt-sources 4 --source-size 128 --depression-strength 0.0 --n-epochs 25 -v

For training the oracle model, please run: 

    python hparam.py --model-name Conv3Net-[Corruption Short Name]_srb-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name cifar10 --n-sources 10 --n-corrupt-sources 4 --source-size 128 --depression-strength 0.0 --n-epochs 25 -v

The oracle model is not implemented for the "No Corruption" case, since here there is no need for an oracle method.


### Training on CIFAR-100

To train the LAP trained model, please run:

    python hparam.py --model-name Conv3Net_100-[Corruption Short Name]-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name cifar100 --n-sources 10 --n-corrupt-sources 2 --source-size 128 --depression-strength 1.0 --strictness 0.8 --lap-n 25 --hold-off 250 --n-epochs 40 -v

Similarly, for standard training:

    python hparam.py --model-name Conv3Net_100-[Corruption Short Name]-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name cifar100 --n-sources 10 --n-corrupt-sources 2 --source-size 128 --depression-strength 0.0 --n-epochs 40 -v

And for the oracle method:
    
    python hparam.py --model-name Conv3Net_100-[Corruption Short Name]_srb-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name cifar100 --n-sources 10 --n-corrupt-sources 2 --source-size 128 --depression-strength 0.0 --n-epochs 40 -v

The oracle model is not implemented for the "No Corruption" case, since here there is no need for an oracle method.

### Training on F-MNIST

To train the LAP trained model, please run:

    python hparam.py --model-name MLP-[Corruption Short Name]-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name fmnist --n-sources 10 --n-corrupt-sources 6 --source-size 200 --depression-strength 1.0 --strictness 0.8 --lap-n 50 --n-epochs 40 -v

Similarly, for standard training:

    python hparam.py --model-name MLP-[Corruption Short Name]-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name fmnist --n-sources 10 --n-corrupt-sources 6 --source-size 200 --depression-strength 0.0 --n-epochs 40 -v

And for the oracle method:
    
    python hparam.py --model-name MLP-[Corruption Short Name]_srb-drstd --seed 2 4 8 16 32 64 128 256 512 1024 --dataset-name fmnist --n-sources 10 --n-corrupt-sources 6 --source-size 200 --depression-strength 0.0 --n-epochs 40 -v


The oracle model is not implemented for the "No Corruption" case, since here there is no need for an oracle method.

### Where are things saved?

A tensorboard file will be saved in the ```runs``` directory, which shows various metrics and information about when depression was applied, how much depression was applied and the metrics on each source.

Whilst training, a figure will be created within the directory ```./outputs/results/```, which shows the training and validation loss, which can be used to track an ongoing experiment.

Pytorch state dicts will be saved in the directory ```./outputs/models/``` which will be accessed by the testing script when measuring the model's performance. Only a completely trained model will be saved here.

## Testing

To test the models, please use the following:


### For CIFAR-10

For testing the LAP trained and standard model, please run:

    python test.py --model-name Conv3Net-[Corruption Short Name]-drstd --dataset-name cifar10 -v 

For training the oracle model, please run: 

    python test.py --model-name Conv3Net-[Corruption Short Name]_srb-drstd --dataset-name cifar10 -v 

The oracle model is not implemented for the "No Corruption" case, since here there is no need for an oracle method.


### For CIFAR-100

For testing the LAP trained and standard model, please run:

    python test.py --model-name Conv3Net_100-[Corruption Short Name]-drstd --dataset-name cifar100 -v 

For training the oracle model, please run: 

    python test.py --model-name Conv3Net_100-[Corruption Short Name]_srb-drstd --dataset-name cifar100 -v 

The oracle model is not implemented for the "No Corruption" case, since here there is no need for an oracle method.



### For F-MNIST

For testing the LAP trained and standard model, please run:

    python test.py --model-name MLP-[Corruption Short Name]-drstd --dataset-name fmnist -v 

For training the oracle model, please run: 

    python test.py --model-name MLP-[Corruption Short Name]_srb-drstd --dataset-name fmnist -v 

The oracle model is not implemented for the "No Corruption" case, since here there is no need for an oracle method.


### For CIFAR-10N (Real-world Data)

For testing the LAP trained and standard model, please run:

    python test_cifarn_different_noise.py

To then graph the results, please follow the ipython notebook ```graph_cifarn_different_noise.ipynb```

### Where are things saved?

This will save the test results in the directory ```./outputs/test_results/```. Each file will be named according the the model and dataset that the results correspond to.


## Graphing

To create a graph showing the results of the plots, you may use tensorboard to visualise the training runs.

Another method, is to use the ```graph.py``` script to plot a box plot of the results on the test data, which will be saved in ```./outputs/graphs/```. This script will also save a formatted table for easy comparisons of models, saved in  ```./outputs/test_results/```.

Simply run:
    
    python graph.py


Make sure that you have run the training and testing scripts before the graphing script is run. If you change any of the commands given here for training, then you will also need to change the mapping dictionary located in the ```graph.py``` script, to ensure that your experiments are correctly labelled.


## Examples

A IPython Notebook is located in this directory, called ```examples.ipynb```, which contains examples on loading data and models and performing testing and training, in case you want to run your own experiments using these models or optimisers.