import subprocess

import pytest


def run_command(command):
    """Helper function to run a shell command and return success status."""
    try:
        subprocess.run(command, shell=True, check=True)
        return True
    except subprocess.CalledProcessError:
        return False


@pytest.mark.parametrize("dataset", ["mmlu", "gsm8k", "qasper"])
def test_eval(dataset):
    """Full run for 3 dataset types with no MCMC compute"""
    STEPS = 0
    BURN_IN = 0
    NUM_CHAINS = 1
    NUM_REPEATS = 1
    EVAL_EXAMPLES = 2

    if dataset == "qasper":  # Multiprocessing not yet enabled for open-ended
        EVAL_PROCESSES = 1
    else:
        EVAL_PROCESSES = 2

    command = (
        f"HYDRA_FULL_ERROR=1 "
        f"python main.py +method=bop +data={dataset} "
        f"method.eval_kwargs.num_processes={EVAL_PROCESSES} "
        f"method.eval_kwargs.num_examples={EVAL_EXAMPLES} "
        f"method.steps={STEPS} method.burn_in={BURN_IN} "
        f"method.num_chains={NUM_CHAINS} method.num_repeats={NUM_REPEATS}"
    )
    assert run_command(command)


def test_chain():
    """Run MH steps"""
    DATASET = "mmlu"
    STEPS = 2
    BURN_IN = 2
    NUM_CHAINS = 1
    NUM_REPEATS = 1
    EVAL_EXAMPLES = 2
    EVAL_PROCESSES = 2

    command = (
        f"HYDRA_FULL_ERROR=1 "
        f"python main.py +method=bop +data={DATASET} "
        f"method.eval_kwargs.num_processes={EVAL_PROCESSES} "
        f"method.eval_kwargs.num_examples={EVAL_EXAMPLES} "
        f"method.steps={STEPS} method.burn_in={BURN_IN} "
        f"method.num_chains={NUM_CHAINS} method.num_repeats={NUM_REPEATS}"
    )
    assert run_command(command)


def test_chain_no_reject():
    DATASET = "mmlu"
    STEPS = 2
    BURN_IN = 2
    NUM_CHAINS = 1
    NUM_REPEATS = 1
    EVAL_EXAMPLES = 2
    EVAL_PROCESSES = 2

    command = (
        f"HYDRA_FULL_ERROR=1 "
        f"python main.py +method=bop method/mcmc=langevin +data={DATASET} "
        f"method.eval_kwargs.num_processes={EVAL_PROCESSES} "
        f"method.eval_kwargs.num_examples={EVAL_EXAMPLES} "
        f"method.steps={STEPS} method.burn_in={BURN_IN} "
        f"method.num_chains={NUM_CHAINS} method.num_repeats={NUM_REPEATS}"
    )
    assert run_command(command)


def test_checkpoint(tmp_path):
    # Quick run that generates checkpoint
    DATASET = "mmlu"
    STEPS = 0
    BURN_IN = 0
    NUM_CHAINS = 1
    NUM_REPEATS = 2
    EVAL_EXAMPLES = 1
    output_dir = tmp_path  # This is a temporary dir unique to this test generated by pytest

    command = (
        f"HYDRA_FULL_ERROR=1 "
        f"python main.py +method=bop method/mcmc=langevin +data={DATASET} "
        f"method.eval_kwargs.num_examples={EVAL_EXAMPLES} "
        f"method.steps={STEPS} method.burn_in={BURN_IN} "
        f"method.num_chains={NUM_CHAINS} method.num_repeats={NUM_REPEATS} "
        f"output_dir={output_dir}"
    )
    assert run_command(command)

    # Another run using checkpoint from first run
    checkpoint_path = output_dir / "ensemble.pkl"

    command = (
        f"HYDRA_FULL_ERROR=1 "
        f"python main.py +method=bop +data={DATASET} "
        f"method.eval_kwargs.num_examples={EVAL_EXAMPLES} "
        f"+checkpoint_path={checkpoint_path}"
    )
    assert run_command(command)
