import os
import numpy as np
import ipdb
import torch
import copy
import torch.nn.functional as F
from torch.optim import Adam
from .utils import soft_update, hard_update
from .model import GaussianPolicy, ValueNetwork, QNetwork, DeterministicPolicy


class BAC(object):
    def __init__(self, num_inputs, action_space, args):


        self.gamma = args.gamma
        self.tau = args.tau
        self.alpha = args.alpha
        self.quantile = args.quantile
        self.lambda_method = args["lambda"]
        self._max_q_grad = 1e-7

        self.policy_type = args.policy
        self.target_update_interval = args.target_update_interval
        self.automatic_entropy_tuning = args.automatic_entropy_tuning

        self.device = torch.device("cuda:{}".format(str(args.device)) if args.cuda else "cpu")
        self._last_q_grad = torch.ones((args.batch_size, 1)).to(device=self.device)

        self.Q_critic = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(device=self.device)
        self.Q_critic_optim = Adam(self.Q_critic.parameters(), lr=args.lr)

        self.V_critic = ValueNetwork(num_inputs, args.hidden_size).to(device=self.device)
        self.V_critic_optim = Adam(self.V_critic.parameters(), lr=args.lr)

        self.Q_critic_target = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(self.device)
        hard_update(self.Q_critic_target, self.Q_critic)

        if self.policy_type == "Gaussian":
            if self.automatic_entropy_tuning is True:
                self.target_entropy = -torch.prod(torch.Tensor(action_space.shape).to(self.device)).item()
                self.log_alpha = torch.zeros(1, requires_grad=True, device=self.device)
                self.alpha_optim = Adam([self.log_alpha], lr=args.lr)

            self.policy = GaussianPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Adam(self.policy.parameters(), lr=args.lr)

        else:
            self.alpha = 0
            self.automatic_entropy_tuning = False
            self.policy = DeterministicPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Adam(self.policy.parameters(), lr=args.lr)

    def select_action(self, state, evaluate=False):
        state = torch.FloatTensor(state).to(self.device).unsqueeze(0)
        if evaluate is False:
            action, _, _ = self.policy.sample(state)
        else:
            _, _, action = self.policy.sample(state)
        return action.detach().cpu().numpy()[0]

    def update_parameters(self, memory, batch_size, updates):
        state_batch, action_batch, reward_batch, next_state_batch, mask_batch = memory.sample(batch_size=batch_size)

        state_batch = torch.FloatTensor(state_batch).to(self.device)
        next_state_batch = torch.FloatTensor(next_state_batch).to(self.device)
        action_batch = torch.FloatTensor(action_batch).to(self.device)
        reward_batch = torch.FloatTensor(reward_batch).to(self.device).unsqueeze(1)
        mask_batch = torch.FloatTensor(mask_batch).to(self.device).unsqueeze(1)

        with torch.no_grad():
            next_state_action, next_state_log_pi, _ = self.policy.sample(next_state_batch)
            qf1_next_target, qf2_next_target = self.Q_critic_target(next_state_batch, next_state_action)
            min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
            next_q_value_e = reward_batch + mask_batch * self.gamma * (min_qf_next_target)  
        qf1, qf2 = self.Q_critic(state_batch, action_batch) 

        target_Vf_pred = self.V_critic(next_state_batch)
        next_q_value_d = reward_batch + mask_batch * self.gamma * target_Vf_pred  

        if self.lambda_method[:6] == "fixed_":
            q_pred_1, q_pred_2 = self.Q_critic_target(state_batch, action_batch)
            q_grad = torch.min(q_pred_1, q_pred_2) - torch.min(qf1, qf2)
            value = float(self.lambda_method[6:])
            lambda_q_d = torch.ones((batch_size, 1)).to(device=self.device)
            lambda_q_d = value * lambda_q_d

        next_q_value = lambda_q_d * next_q_value_d + (1 - lambda_q_d) * next_q_value_e

        qf1_loss = F.mse_loss(qf1, next_q_value)
        qf2_loss = F.mse_loss(qf2, next_q_value)
        qf_loss = qf1_loss + qf2_loss

        
        pi, log_pi, _ = self.policy.sample(state_batch)
        vf_pred = self.V_critic(state_batch) - self.alpha * log_pi
        q_pred_1,q_pred_2 = self.Q_critic_target(state_batch, action_batch)
        q_pred = torch.min(q_pred_1, q_pred_2)


        vf_err = q_pred - vf_pred
        vf_sign = (vf_err < 0).float()
        vf_weight = (1 - vf_sign) * self.quantile + vf_sign * (1 - self.quantile)
        vf_loss = (vf_weight * (vf_err ** 2)).mean()

        self.Q_critic_optim.zero_grad()
        qf_loss.backward()
        self.Q_critic_optim.step()

        self.V_critic_optim.zero_grad()
        vf_loss.backward()
        self.V_critic_optim.step()

        # compute policy loss
        pi, log_pi, _ = self.policy.sample(state_batch)
        qf1_pi, qf2_pi = self.Q_critic(state_batch, pi)
        min_qf_pi = torch.min(qf1_pi, qf2_pi)

        if torch.is_tensor(self.alpha):
            policy_loss = ((self.alpha.detach() * log_pi) - min_qf_pi).mean() # Jπ = 𝔼st∼D,εt∼N[α * logπ(f(εt;st)|st) − Q(st,f(εt;st))]
        else:
            policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean()
        self.policy_optim.zero_grad()
        policy_loss.backward()
        self.policy_optim.step()

        if self.automatic_entropy_tuning:
            alpha_loss = -(self.log_alpha * (log_pi + self.target_entropy).detach()).mean()

            self.alpha_optim.zero_grad()
            alpha_loss.backward()
            self.alpha_optim.step()

            self.alpha = self.log_alpha.exp()
            alpha_tlogs = self.alpha.clone() 
        else:
            alpha_loss = torch.tensor(0.).to(self.device)
            alpha_tlogs = torch.tensor(self.alpha) 


        if updates % self.target_update_interval == 0:
            soft_update(self.Q_critic_target, self.Q_critic, self.tau)

        return qf1_loss.item(), qf2_loss.item(), vf_loss.item(), policy_loss.item(), alpha_loss.item(), alpha_tlogs.item(), torch.mean(q_grad), torch.mean(lambda_q_d), torch.mean(next_q_value), torch.mean(next_q_value_e), torch.mean(next_q_value_d)

    def save_checkpoint(self, path, i_episode):
        ckpt_path = path + '/' + '{}.torch'.format(i_episode)
        print('Saving models to {}'.format(ckpt_path))
        torch.save({'policy_state_dict': self.policy.state_dict(),
                    'critic_state_dict': self.Q_critic.state_dict(),
                    'critic_target_state_dict': self.Q_critic_target.state_dict(),
                    'value_state_dict': self.V_critic.state_dict(),
                    'critic_optimizer_state_dict': self.Q_critic_optim.state_dict(),
                    'policy_optimizer_state_dict': self.policy_optim.state_dict(),
                    'value_optimizer_state_dict': self.V_critic_optim.state_dict()}, ckpt_path)
    
    def load_checkpoint(self, path, i_episode, evaluate=False):
        ckpt_path = path + '/' + '{}.torch'.format(i_episode)
        print('Loading models from {}'.format(ckpt_path))
        if ckpt_path is not None:
            checkpoint = torch.load(ckpt_path)
            self.policy.load_state_dict(checkpoint['policy_state_dict'])
            self.Q_critic.load_state_dict(checkpoint['critic_state_dict'])
            self.Q_critic_target.load_state_dict(checkpoint['critic_target_state_dict'])
            self.V_critic.load_state_dict(checkpoint['value_state_dict'])
            self.Q_critic_optim.load_state_dict(checkpoint['critic_optimizer_state_dict'])
            self.policy_optim.load_state_dict(checkpoint['policy_optimizer_state_dict'])
            self.V_critic_optim.load_state_dict(checkpoint['value_optimizer_state_dict'])

            if evaluate:
                self.policy.eval()
                self.Q_critic.eval()
                self.Q_critic_target.eval()
                self.V_critic.eval()
            else:
                self.policy.train()
                self.Q_critic.train()
                self.Q_critic_target.train()
                self.V_critic.train()
