import os
import random
import argparse
import logging
import inspect

import yaml
import torch
import numpy as np
from dotmap import DotMap

from src.tamp import ITDiff
from utils.log import Logger

def main():

    parser = argparse.ArgumentParser(prog='The main entry')
    parser.add_argument('-c', '--config', type=str, default='./config/default_config.yml', help='model configuration (default: ./config/default_config.yml))')
    args = parser.parse_args()

    if not os.path.isfile(args.config):
        raise FileExistsError("The provided configuration is not a file or does not exist!")
    with open(args.config, 'r') as cf:
        try:
            cfg = yaml.safe_load(cf)
        except yaml.YAMLError as exc:
            print(exc)
    cfg = DotMap(cfg)

    # setup logger
    logger = Logger(
        log_path=os.path.join(cfg.SAVE_DEST, cfg.EXP_NAME, cfg.SETUP_NAME),
        log_file='log.txt'
    )
    logger.log('Logging to {}!'.format(logger._log_dest))

    # init device
    if torch.cuda.is_available():
        if cfg.GPU is None:
            logger.log("CUDA is available but no gpu specified, default device is utilized.")
            cfg.DEVICE = "cuda:0"
        else:
            assert isinstance(cfg.GPU, int), 'The GPU should be specified with an int!'
            device_pool = [i for i in range(torch.cuda.device_count())]
            assert cfg.GPU in device_pool, \
                'GPU: %s not found'%cfg.GPU
            cfg.DEVICE = "cuda:%d"%cfg.GPU
        torch.backends.cudnn.benchmark = True   # cudnn auto-tuner
    else:
        cfg.DEVICE = "cpu"
        logger.log("CUDA is not available, cpu is utilized.")

    # set cv2 running threads to 1 (prevents deadlocks with pytorch dataloader)
    # cv2.setNumThreads(0)

    # initialize random seed
    if not cfg.SEED:
        cfg.SEED = random.randint(0, 10000)
    logger.log("Setting random seed as: {}".format(cfg.SEED))
    random.seed(cfg.SEED)
    np.random.seed(cfg.SEED)
    torch.manual_seed(cfg.SEED)
    if torch.cuda.is_available():
        torch.cuda.manual_seed_all(cfg.SEED)

    # path works
    cfg.CHECKPOINT_DEST = os.path.join(cfg.SAVE_DEST, cfg.EXP_NAME, cfg.SETUP_NAME, 'checkpoint')
    if not os.path.exists(cfg.CHECKPOINT_DEST):
        os.makedirs(cfg.CHECKPOINT_DEST)
    cfg.SAMPLE_DEST = os.path.join(cfg.SAVE_DEST, cfg.EXP_NAME, cfg.SETUP_NAME, 'samples')
    if not os.path.exists(cfg.SAMPLE_DEST):
        os.makedirs(cfg.SAMPLE_DEST)
    cfg.PRETRAIN_PATH = os.path.join(os.getcwd(), 'checkpoint')

    # save configuration
    config_file = os.path.join(cfg.SAVE_DEST, cfg.EXP_NAME, cfg.SETUP_NAME, 'config.yaml')
    with open(config_file, 'w') as f:
        yaml.dump(cfg.toDict(), f, default_flow_style=False)

    # running
    assert cfg.MODE in ['train', 'test', 'eval'], \
        'Unsupported experimental mode: {}'.format(cfg.MODE)

    model = ITDiff(cfg, logger)
    model.load_diffusion()
    # do not load misf when train model without resume
    if not(cfg.MODE == 'train' and not cfg.RESUME):
        model.load_misf(amend=False)

    func_name = cfg.MODE + '_in_ddnm_style'
    assert func_name in [func[0] for func in inspect.getmembers(ITDiff, predicate=inspect.isfunction)]

    logger.log('Start {}...\n'.format(func_name))
    getattr(model, func_name)()


if __name__ == "__main__":
    main()
