from pathlib import Path
from typing import Literal, Optional, Self, Union

from redacted import (
    ChatModel,
    ChatRequest,
    ChatResponse,
    JsonCache,
    Message,
    ModelSupports,
    Client,
    Models,
)
from pydantic import BaseModel, model_validator


class ModelConfiguration(BaseModel):
    name: str
    n: int = 1
    temperature: Optional[float] = None
    reasoning: Optional[Literal["low", "medium", "high"]] = None

    @property
    def specification(self) -> dict[str, Union[str, int, float]]:
        return Models.by_name(self.name)

    @property
    def request(self) -> ChatRequest:
        return ChatRequest(
            n=self.n, temperature=self.temperature, reasoning_effort=self.reasoning
        )

    @property
    def name(self) -> str:
        if self.reasoning:
            return f"{self.name}-{self.reasoning}"
        return self.name

    @model_validator(mode="after")
    def validate(self) -> Self:
        if self.name not in Models.names():
            raise ValueError(f"Unknown model name: {self.name}")
        specification = Models.by_name(self.name)
        reasoning = ModelSupports.Reasoning in specification.supports
        if reasoning:
            self.temperature = None
            if self.reasoning is None:
                self.reasoning = "low"
        else:
            if self.temperature is None:
                self.temperature = 0.0
        return self


class ConfiguredModel:
    """A model configured with specific settings."""

    def __init__(self, configuration: ModelConfiguration, cache: Optional[Path] = None):
        self.configuration = configuration
        self.model = ChatModel(
            model=configuration.specification, client=Client()
        )
        if cache is not None:
            self.model.cache = JsonCache(cache, autosave=4, write=True)

    def chat(self, messages: list[Message]) -> ChatResponse:
        return self.model.chat(messages, self.configuration.request)
