from argparse import ArgumentParser
from pathlib import Path
import json
import os
import subprocess
from tqdm.auto import tqdm
from repo_graph import RepoGraph

from typing import Union

from ..utils import ContextManager


def main(
    repo_path: Union[str, Path],
    task_instances_path: Union[str, Path],
    output_dir: Union[str, Path],
):
    repo_path = Path(repo_path).resolve().absolute()
    output_dir = Path(output_dir).resolve().absolute()
    if not output_dir.exists():
        output_dir.mkdir(parents=True)
    with open(task_instances_path, "r") as f:
        instances = [json.loads(line) for line in f]
        instances = sorted(instances, key=lambda x: x["created_at"])
    for instance in tqdm(instances):
        output_filename = output_dir / f"{instance['instance_id']}.repo_graph.json"
        if output_filename.exists():
            continue
        with ContextManager(repo_path, instance["base_commit"], verbose=False):
            repo_graph = RepoGraph(repo_path)
            data = repo_graph.to_json()
            with open(output_filename, "w") as f:
                f.write(data)


if __name__ == "__main__":
    parser = ArgumentParser()
    parser.add_argument(
        "repo_path",
        type=str,
        help="Path to the repository.",
    )
    parser.add_argument(
        "task_instances_path",
        type=str,
        help="Path to the task instances JSONL file.",
    )
    parser.add_argument(
        "output_dir",
        type=str,
        help="Path to the output directory.",
    )
    args = parser.parse_args()
    main(**vars(args))
