import numpy as np


def nice_logspace(start, end, base=10, density=0):
    """Returns a log-spaced grid between ``base**start`` and ``base**end``.
    Increasing the density repeats previous values.

    - ``Start``, ``end`` and ``density`` are integers.
    - Increasing ``density`` by 1 approximately doubles the number of points
    - ``Density = 1`` will return ``end - start + 1`` points
    - ``nice_logspace(0, 1, base=10, density=1) == [1, 10] == [10**0, 10**1]``
    - ``Density = 2`` will return ``2*(end-start) + 1`` points
    - ``nice_logspace(0, 1, base=10, density=2) == [1, 3.16..., 10] == [10**0, 10**(1/2), 10**1]``
    """
    if density < 0 or not np.allclose(int(density), density):
        raise ValueError(f"Density needs to be an integer >= 0, got {density}.")
    if not np.allclose(int(start), start) or not np.allclose(int(end), end):
        raise ValueError(f"Start and end need to be integers, got {start, end}.")
    start, end, density = int(start), int(end), int(density)
    return np.logspace(start, end, base=base, num=(end - start) * (2**density) + 1)


def flatten(l):
    """Flattens a list of lists."""
    return [item for sublist in l for item in sublist]
