import os
from pathlib import Path

import git
import matplotlib as mpl
import matplotlib.pyplot as plt
from omegaconf import OmegaConf

from graphsmodel.utils import decode_path, get_project_name, get_results_dir, replace

try:
    PROJECT_ROOT = Path(
        git.Repo(Path.cwd(), search_parent_directories=True).working_dir
    )
except git.exc.InvalidGitRepositoryError:
    PROJECT_ROOT = Path.cwd()

os.environ["PROJECT_ROOT"] = str(PROJECT_ROOT)
os.environ["WANDB_CACHE_DIR"] = str(PROJECT_ROOT / ".wandb_cache")

try:

    OmegaConf.register_new_resolver("path", lambda path: decode_path(path))
    OmegaConf.register_new_resolver("if", lambda c, x, y: x if c else y)
    OmegaConf.register_new_resolver(
        "replace",
        lambda name, to_replace, replace_with: replace(name, to_replace, replace_with),
    )
    OmegaConf.register_new_resolver(
        "get_project_name",
        lambda project_name, model_target, dataset_name: get_project_name(
            project_name, model_target, dataset_name
        ),
    )
    OmegaConf.register_new_resolver(
        "get_results_dir",
        lambda storage_dir, dataset_name, model_name, data_seed, subsets_seed, subset_mode, task_induced_subgraph,: get_results_dir(
            storage_dir,
            dataset_name,
            model_name,
            data_seed,
            subsets_seed,
            subset_mode,
            task_induced_subgraph,
        ),
    )
except ValueError:
    pass

mpl.rcParams["axes.formatter.use_mathtext"] = True
mpl.rcParams["axes.formatter.limits"] = (-3, 3)

# Define global settings
plt.rcParams.update(
    {
        "figure.figsize": (3, 3),  # Figure size
        "figure.dpi": 300,  # Dots per inch (resolution)
        "font.size": 10,  # Font size
        "axes.grid": True,  # Show grid lines by default
        "grid.linestyle": "--",  # Grid line style
        "grid.alpha": 0.7,  # Grid line transparency
        "grid.linewidth": 0.7,  # Grid line width
        "legend.fontsize": 6,  # Legend font size
        "lines.markersize": 0.5,  # Marker size
        "lines.linewidth": 0.7,  # Line width
        "axes.linewidth": 0.8,  # Axis width
        "axes.labelpad": 2,  # Padding between axis and axis labels
        "xtick.major.pad": 0,  # Padding between x-ticks and x-axis
        "xtick.labelsize": 6,  # X-tick label size
        "ytick.major.pad": 0,  # Padding between y-ticks and y-axis
        "ytick.labelsize": 6,  # Y-tick label size
        "xtick.bottom": False,  # ticks
        "xtick.top": False,
        "ytick.left": False,
        "ytick.right": False,
        "figure.constrained_layout.use": True,
    }
)

# Set the desired width-to-height aspect ratio (e.g., 16:9)
aspect_ratio = 1  # For square subplots, set aspect_ratio to 1

# Calculate the individual subplot size based on the aspect ratio
SUBPLOT_WIDTH = 2  # Adjust this value as needed
SUBPLOT_HEIGHT = SUBPLOT_WIDTH / aspect_ratio

palette = {
    "tail:blue": "#99ccff",
    "tail:orange": "#ffcc80",
    "tail:teal": "#3b9a9c",
    "tail:pink": "#ff99cc",
    "tail:yellow": "#ffff99",
    "tail:skyblue": "#80dfff",
    "tail:green": "#99e699",
    "tail:gold": "#ffd700",
    "tail:red": "#ff6666",
    "tail:grey": "#cccccc",
    "tail:lime": "#ccff99",
    "tail:purple": "#c099ff",
    "tail:magenta": "#ff66ff",
    "tail:cyan": "#66ffe6",
    "tail:brown": "#ac7339",
    "tail:maroon": "#800000",
    "tail:navy": "#004c99",
    "tail:olive": "#99cc66",
    "tail:peach": "#ffcc99",
    "tail:aqua": "#7fffd4",
}
