# run_skeleton_task.py (Filled In for Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All available skills—do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # E.g. object_poses = {'drawer1': (x,y,z), ...}
        positions = get_object_positions()

        # === EXPLORATION PHASE: Identify Missing Predicate via Skills ===
        # Since the planner timed out, there is likely a missing predicate in the domain or problem description that prevents progress.
        # The purpose here is to actively execute available skills in the environment to gather more information for domain debugging.

        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Example logic: iterate objects, try each skill, detect if an action "reveals" a previously unmodeled predicate

        # Get all objects and locations (using observation or environment conventions)
        try:
            # This should be dynamically gathered from the environment or initial observation parse
            all_objects = list(positions.keys())
        except Exception as e:
            print("[Exploration] Could not get object keys from positions:", e)
            all_objects = []

        # For demonstration, select some objects and locations if present
        # (Below, dummy fallback names as illustration)
        example_objects = all_objects if all_objects else ['object1', 'object2']
        example_locations = ['ready-pose', 'table', 'drawer']
        found_missing_predicate = False

        print("[Exploration] Starting active predicate investigation using available skills")

        # Loop through available skills and objects to probe for missing predicate evidence
        for obj_name in example_objects:
            obj_pos = positions.get(obj_name, None)

            # Test pick skill
            try:
                print(f"[Exploration] Attempting execute_pick on {obj_name} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=obj_name,
                    target_pos=obj_pos,
                    approach_distance=0.12,
                    max_steps=30,
                    threshold=0.015,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Pick successful, no missing precondition for pick on", obj_name)
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick on {obj_name}: {e}")
                # If error refers to a missing predicate, log it
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_pick` detected")
                    found_missing_predicate = True

            # Test place skill
            try:
                print(f"[Exploration] Attempting execute_place with {obj_name}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    target_object=obj_name,
                    target_drawer='drawer',  # Assuming 'drawer' is present
                    target_pos=obj_pos,
                    approach_distance=0.12,
                    max_steps=40,
                    threshold=0.015,
                    approach_axis='z',
                    timeout=7.0
                )
                if done:
                    print("[Exploration] Place successful")
            except Exception as e:
                print(f"[Exploration] Exception during execute_place on {obj_name}: {e}")
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_place` detected")
                    found_missing_predicate = True

            # Test push, pull, sweep, rotate, gripper (where reasonable)
            try:
                print(f"[Exploration] Attempting execute_push for {obj_name}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    target_drawer=obj_name,  # Try pushing as if it is a drawer (may fail if wrong type)
                    target_pos=obj_pos,
                    approach_distance=0.10,
                    max_steps=20,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=4.0
                )
            except Exception as e:
                print(f"[Exploration] Exception during execute_push on {obj_name}: {e}")
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_push` detected")
                    found_missing_predicate = True

            try:
                print(f"[Exploration] Attempting execute_pull for {obj_name}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    target_drawer=obj_name,  # Try pulling as if it is a drawer (may fail if not a handle)
                    target_handle=obj_name,
                    target_pos=obj_pos,
                    approach_distance=0.10,
                    max_steps=20,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=4.0
                )
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull on {obj_name}: {e}")
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_pull` detected")
                    found_missing_predicate = True

            try:
                print(f"[Exploration] Attempting execute_sweep for {obj_name}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    target_object=obj_name,
                    target_pos=obj_pos,
                    approach_distance=0.10,
                    max_steps=20,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=4.0
                )
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep on {obj_name}: {e}")
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_sweep` detected")
                    found_missing_predicate = True

            try:
                print(f"[Exploration] Attempting execute_rotate for {obj_name}")
                obs, reward, done = execute_rotate(
                    env,
                    task,
                    target_object=obj_name,
                    target_pos=obj_pos,
                    angle=90,
                    approach_distance=0.12,
                    max_steps=25,
                    threshold=0.017,
                    approach_axis='z',
                    timeout=6.0
                )
            except Exception as e:
                print(f"[Exploration] Exception during execute_rotate on {obj_name}: {e}")
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_rotate` detected")
                    found_missing_predicate = True

            try:
                print(f"[Exploration] Attempting execute_gripper")
                obs, reward, done = execute_gripper(
                    env,
                    task
                )
            except Exception as e:
                print(f"[Exploration] Exception during execute_gripper: {e}")
                if 'predicate' in str(e).lower():
                    print("[Exploration] Potential missing predicate for `execute_gripper` detected")
                    found_missing_predicate = True

        # Test navigation skills between locations if relevant
        for from_loc in example_locations:
            for to_loc in example_locations:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Attempting execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=from_loc,
                            to_location=to_loc,
                            approach_distance=0.10,
                            max_steps=40,
                            threshold=0.02,
                            timeout=8.0
                        )
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_go ({from_loc}->{to_loc}): {e}")
                        if 'predicate' in str(e).lower():
                            print("[Exploration] Potential missing predicate for `execute_go` detected")
                            found_missing_predicate = True

        if found_missing_predicate:
            print("[Exploration] Detected evidence for missing predicate via skill execution. Check above logs for candidate predicates.")
        else:
            print("[Exploration] No missing predicate detected during skill probing. Please review the domain/problem encoding.")

        # If further plan steps are available, they should be executed in the next phase (not part of exploration)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
