# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine skills, use them as imported

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration phase to determine missing predicate by executing all available skills ===
        print("[Exploration] Starting predicate exploration using available skills...")
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]
        
        # Try all skills in sequence with safe parameters
        # We must not create new skill functions; parameters must be obtained from the env/state
        # We'll attempt to call each skill, catching any errors, and log their effects
        
        # Helper: get all objects, locations, handles, and drawers, if possible
        try:
            object_names = [name for name in positions.keys() if "obj" in name or "ball" in name or "object" in name]
        except Exception:
            object_names = []
        # Fallback: generic test names if positions are not detailed enough
        if not object_names:
            object_names = ["object_1", "object_2", "drawer_1", "handle_1"]

        # Get possible locations (assume key format 'location_x' or fallback)
        location_names = [name for name in positions.keys() if "location" in name or "room" in name or "pose" in name]
        if not location_names:
            location_names = ["ready-pose", "location_1", "location_2"]

        # Get possible handles and drawers by direct inspection of names
        handle_names = [name for name in positions.keys() if "handle" in name]
        drawer_names = [name for name in positions.keys() if "drawer" in name]
        if not handle_names:
            handle_names = ["handle_1"]
        if not drawer_names:
            drawer_names = ["drawer_1"]
        
        # Robot initial location for move actions
        robot_initial_loc = "ready-pose"
        if "robot" in positions.keys():
            robot_initial_loc = positions.get("robot")

        # For exploration, we try minimal, non-destructive invocations of the skills
        # Try execute_pick (assuming pick handles and pick objects if exists)
        for obj_name in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj_name} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj_name, loc)
                except Exception as e:
                    print(f"execute_pick failed for {obj_name}, {loc}: {e}")
        # Try execute_place (assuming 'object', 'drawer', 'location')
        for obj_name in object_names:
            for dr in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_place: {obj_name} into {dr} at {loc}")
                        obs, reward, done = execute_place(env, task, obj_name, dr, loc)
                    except Exception as e:
                        print(f"execute_place failed for {obj_name}, {dr}, {loc}: {e}")
        # Try execute_push (drawer, location)
        for dr in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push: {dr} at {loc}")
                    obs, reward, done = execute_push(env, task, dr, loc)
                except Exception as e:
                    print(f"execute_push failed for {dr}, {loc}: {e}")
        # Try execute_pull (drawer, handle, location)
        for dr in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_pull: {dr} with {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, dr, handle, loc)
                    except Exception as e:
                        print(f"execute_pull failed for {dr}, {handle}, {loc}: {e}")
        # Try execute_sweep (object, location)
        for obj_name in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_sweep: {obj_name} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj_name, loc)
                except Exception as e:
                    print(f"execute_sweep failed for {obj_name}, {loc}: {e}")
        # Try execute_rotate (object, location) if exists
        if 'execute_rotate' in available_skills:
            for obj_name in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_rotate: {obj_name} at {loc}")
                        obs, reward, done = execute_rotate(env, task, obj_name, loc)
                    except Exception as e:
                        print(f"execute_rotate failed for {obj_name}, {loc}: {e}")
        # Try execute_go (from, to)
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go: {from_loc} -> {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    except Exception as e:
                        print(f"execute_go failed for {from_loc} -> {to_loc}: {e}")
        # Try execute_gripper (no arguments)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check output logs for skill preconditions/effects and missing predicate information.")

        # Next, based on feedback from these attempts (such as error outputs or failed preconditions), user or automated agent can identify missing predicates.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()