# run_skeleton_task.py (Fully Generic Skeleton, completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # predefined skills only, do not redefine!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ======== EXPLORATION PHASE =========
        # Feedback indicates a missing predicate: (need-ready)
        # Exploration required to check for 'need-ready' requirements in action preconditions.

        print("[Exploration] Investigating possible missing predicate through sequential skill testing.")

        # We'll try executing the main skill sequence, and if an action fails (skipped due to 'need-ready'),
        # we'll attempt to clear 'need-ready' with execute_go_ready before retrying the intended skill.

        # EXAMPLE ORACLE PLAN (pseudo-code, should be generated from planner; here is a generic exploration sequence):
        # For demonstration, suppose the flow is (pick object A, place in drawer B, push drawer B), which, according to
        # PDDL, will set 'need-ready' after pick/place, requiring go_ready to clear before next skill.

        # The below object/drawer/location names are placeholders—you would extract these from the provided descriptions/objects/init.
        # For exploration, we'll get *some* object/drawer/location names from positions dict.
        # Use try/except to avoid hard crashes!

        # Get candidate objects and locations
        object_names = [k for k in positions.keys() if 'object' in k or 'obj' in k]
        drawer_names = [k for k in positions.keys() if 'drawer' in k]
        handle_names = [k for k in positions.keys() if 'handle' in k]
        loc_names = [k for k in positions.keys() if 'loc' in k or 'pose' in k or 'floor' in k or 'room' in k]
        # fallback, just use all available as locations
        if not loc_names:
            loc_names = list(positions.keys())

        # Pick representative names if sets are not empty
        obj = object_names[0] if object_names else None
        drw = drawer_names[0] if drawer_names else None
        hdl = handle_names[0] if handle_names else None
        loc = loc_names[0] if loc_names else None
        # For ready-pose, see if available, else use a default/fallback
        ready_pose = 'ready-pose' if 'ready-pose' in positions else loc_names[0] if loc_names else None

        # --------------- Exploration sequence ---------------
        # Try a pick->place->push sequence,
        # after each action that sets 'need-ready', call execute_go_ready

        try:
            # 1. Move to obj location if needed (simulate with execute_go)
            print("[Exploration] Moving to the object's location.")
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=loc)
            if done:
                print("[Exploration] Early termination during move.")
                return

            # 2. Try to pick up the object: will set need-ready
            print("[Exploration] Attempting to pick the object. If blocked by need-ready, will clear with go_ready.")
            try:
                obs, reward, done = execute_pick(env, task, obj=obj, location=loc)
            except Exception as e:
                print("[Exploration] Exception during execute_pick:", e)
                # If action is blocked, try clearing with execute_go_ready and retry
                print("[Exploration] Attempting to clear need-ready using execute_go_ready.")
                obs, reward, done = execute_go_ready(env, task, from_location=loc)
                print("[Exploration] Retrying execute_pick.")
                obs, reward, done = execute_pick(env, task, obj=obj, location=loc)
            if done:
                print("[Exploration] Early termination after pick.")
                return

            # 3. After pick, must execute_go_ready due to need-ready
            print("[Exploration] Executing go_ready after picking (to clear need-ready).")
            try:
                obs, reward, done = execute_go_ready(env, task, from_location=loc)
            except Exception as e:
                print("[Exploration] Exception in execute_go_ready:", e)
            if done:
                print("[Exploration] Early termination after go_ready.")
                return

            # 4. Move to drawer/destination location if needed (simulate with execute_go)
            if drw:
                print("[Exploration] Moving to drawer location for placement.")
                obs, reward, done = execute_go(env, task, from_location=loc, to_location=drw)
                if done:
                    print("[Exploration] Early termination during move to drawer.")
                    return
            else:
                print("[Exploration] Drawer not found in positions, skipping place.")

            # 5. Try to place the object in the drawer
            print("[Exploration] Placing the object in the drawer. If blocked by need-ready, will clear.")
            try:
                obs, reward, done = execute_place(env, task, obj=obj, drawer=drw, location=drw)
            except Exception as e:
                print("[Exploration] Exception during execute_place:", e)
                print("[Exploration] Attempting to clear need-ready using execute_go_ready.")
                obs, reward, done = execute_go_ready(env, task, from_location=drw)
                print("[Exploration] Retrying execute_place.")
                obs, reward, done = execute_place(env, task, obj=obj, drawer=drw, location=drw)
            if done:
                print("[Exploration] Early termination after place.")
                return

            # 6. After placing, clear need-ready again
            print("[Exploration] Executing go_ready after placing (to clear need-ready).")
            try:
                obs, reward, done = execute_go_ready(env, task, from_location=drw)
            except Exception as e:
                print("[Exploration] Exception in execute_go_ready:", e)
            if done:
                print("[Exploration] Early termination after go_ready.")
                return

            # 7. (Optional) Try push drawer (assuming correct precondition: drawer open, robot-free, at drawer)
            if drw:
                print("[Exploration] Attempting to push the drawer.")
                try:
                    obs, reward, done = execute_push(env, task, drawer=drw, location=drw)
                except Exception as e:
                    print("[Exploration] Exception during execute_push:", e)
                    # Sometimes push might be blocked due to missing clear or predicate
                    print("[Exploration] Attempting to clear need-ready using execute_go_ready before push.")
                    obs, reward, done = execute_go_ready(env, task, from_location=drw)
                    print("[Exploration] Retrying execute_push.")
                    obs, reward, done = execute_push(env, task, drawer=drw, location=drw)
                if done:
                    print("[Exploration] Task complete after push.")
                    return

            print("[Exploration] Exploration sequence complete. If errors required execute_go_ready to clear 'need-ready', the missing predicate is confirmed.")

        except Exception as e:
            print("[ERROR] Unexpected exception during exploration:", e)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")
        

if __name__ == "__main__":
    run_skeleton_task()