# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from feedback and example) ===
        object_list = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2",
            "trash",
            "trash_bin"
        ]

        dice_names = ["dice1", "dice2"]
        drawer_places = ["drawer_top_place_left", "drawer_top_place_right",
                         "drawer_middle_place", "drawer_bottom_place_left", "drawer_bottom_place_right"]

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for object properties (e.g., durability, weight, lock-known)
        # We'll use the available skills to explore and check for missing predicates before manipulation.

        # For each dice, check if it is safe to pick and place (simulate exploration)
        for dice in dice_names:
            try:
                dice_pos = positions[dice]
            except KeyError:
                print(f"[Exploration] {dice} not found in positions. Skipping.")
                continue

            # 1. Move to dice location (if not already there)
            try:
                robot_pos = positions.get('robot', None)
                if robot_pos is not None and not np.allclose(robot_pos, dice_pos, atol=0.05):
                    # Use execute_go to move
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_pos,
                        to_location=dice_pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended while moving to {dice}.")
                        return
            except Exception as e:
                print(f"[Exploration] Error during move to {dice}: {e}")

            # 2. Try to pick the dice and check for durability/weight (simulate exploration)
            # We'll use execute_pick (or execute_pick_object) as per domain
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target=dice,
                    location=dice_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {dice}.")
                    return
            except Exception as e:
                print(f"[Exploration] Error during pick of {dice}: {e}")

            # 3. (Optional) Check for durability/weight/lock-known if such feedback is available
            # In a real system, here we would check obs or task state for missing predicates

            # 4. Place dice in the correct drawer place (safety check)
            # For this example, dice1 -> drawer_top_place_left, dice2 -> drawer_top_place_right
            if dice == "dice1":
                target_drawer = "drawer_top_place_left"
            elif dice == "dice2":
                target_drawer = "drawer_top_place_right"
            else:
                print(f"[Exploration] Unknown dice: {dice}. Skipping placement.")
                continue

            try:
                drawer_pos = positions[target_drawer]
            except KeyError:
                print(f"[Exploration] Drawer place {target_drawer} not found. Skipping placement for {dice}.")
                continue

            # Safety check: is the drawer open? (simulate by checking positions or task state if available)
            # For now, assume the drawer is open or open it if needed
            # (In a real system, would check predicates or use execute_pull/execute_push as needed)

            # Place the dice
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    target=dice,
                    drawer=target_drawer,
                    location=drawer_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {dice}.")
                    return
                print(f"[Exploration] Successfully placed {dice} in {target_drawer}.")
            except Exception as e:
                print(f"[Exploration] Error during place of {dice}: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Completed exploration and placement for all dice.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
