# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates we must check if an object is in the object list before acting.
        # We'll perform a check and log an error if the object is missing.
        # For demonstration, let's assume we want to manipulate 'dice' and put it into a 'drawer'.
        # We'll use only the provided skills and not define new ones.

        # Example object and drawer names (replace with actual names from your environment)
        object_list = list(positions.keys())
        target_object = 'dice'
        target_drawer = 'drawer'
        target_handle = 'handle'
        ready_pose = 'ready-pose'
        robot_location = None

        # Find the robot's current location if available
        for obj_name, pos in positions.items():
            if 'robot' in obj_name:
                robot_location = obj_name
                break

        # Check if the target object exists
        if target_object not in object_list:
            print(f"Error: {target_object} is not in the object list.")
            # Optionally, abort or skip further actions
        else:
            # Additional safety: check for empty object list
            if not object_list:
                print("Error: The object list is empty.")
            else:
                # Optionally, check if the object is a dice (if type info is available)
                # For now, proceed with the plan

                # === Example Plan Execution ===
                try:
                    # 1. Move robot to the object's location if not already there
                    object_pos = positions[target_object]
                    # Assume we have a mapping from object to location name
                    object_location = None
                    for loc_name, pos in positions.items():
                        if loc_name == target_object:
                            object_location = loc_name
                            break
                    if object_location is None:
                        print(f"Error: Could not determine location for {target_object}.")
                    else:
                        # If robot location is known and different, move
                        if robot_location is not None and robot_location != object_location:
                            obs, reward, done = execute_go(env, task, robot_location, object_location)
                            robot_location = object_location

                        # 2. Pick up the object (dice)
                        obs, reward, done = execute_pick(env, task, target_object, object_location)
                        # 3. Move to the drawer location
                        if target_drawer in positions:
                            drawer_location = target_drawer
                        else:
                            # Fallback: use a default or log error
                            print(f"Warning: {target_drawer} not found in positions, using default location.")
                            drawer_location = object_location  # fallback

                        if robot_location != drawer_location:
                            obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                            robot_location = drawer_location

                        # 4. Place the object into the drawer
                        obs, reward, done = execute_place(env, task, target_object, target_drawer, drawer_location)

                        # 5. Return to ready pose if required
                        obs, reward, done = execute_go(env, task, robot_location, ready_pose)
                        robot_location = ready_pose

                except Exception as e:
                    print(f"Exception during plan execution: {e}")

        # === End of Exploration Phase ===
        # The above logic ensures that we check for object existence before acting,
        # as required by the feedback, and only use predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
