# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from feedback and PDDL) ===
        # Objects: dice1, dice2, handle_bottom, handle_middle, handle_top
        # Drawers: drawer_bottom, drawer_middle, drawer_top
        # Locations: floor, drawer-area, ready-pose

        # For exploration, we need to check for missing predicates (e.g., lock-known, identified, etc.)
        # We'll attempt to use available skills to interact with all relevant objects and locations.

        # List of objects and handles
        object_list = [
            "dice1", "dice2",
            "handle_bottom", "handle_middle", "handle_top"
        ]
        drawer_list = [
            "drawer_bottom", "drawer_middle", "drawer_top"
        ]
        handle_list = [
            "handle_bottom", "handle_middle", "handle_top"
        ]
        location_list = [
            "floor", "drawer-area", "ready-pose"
        ]

        # === Exploration Phase: Try available skills to discover missing predicates ===
        # The goal is to interact with all objects and handles using the available skills,
        # and observe if any action fails due to a missing predicate.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to pick up all objects and handles from the floor
        for obj in object_list:
            try:
                print(f"[Exploration] Attempting to pick: {obj} at floor")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=obj,
                    location="floor"
                )
                if done:
                    print(f"[Exploration] Picked {obj} successfully.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj}: {e}")

        # 2. Try to pull each drawer using its handle
        for drawer, handle in zip(drawer_list, handle_list):
            try:
                print(f"[Exploration] Attempting to pick handle: {handle} at floor")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=handle,
                    location="floor"
                )
                if done:
                    print(f"[Exploration] Picked handle {handle} successfully.")
                print(f"[Exploration] Attempting to pull {drawer} with {handle}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer,
                    handle=handle,
                    location="drawer-area"
                )
                if done:
                    print(f"[Exploration] Pulled {drawer} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to pull {drawer} with {handle}: {e}")

        # 3. Try to place dice into drawers
        for obj, drawer in zip(["dice1", "dice2"], ["drawer_bottom", "drawer_top"]):
            try:
                print(f"[Exploration] Attempting to pick: {obj} at floor")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=obj,
                    location="floor"
                )
                if done:
                    print(f"[Exploration] Picked {obj} successfully.")
                print(f"[Exploration] Attempting to place {obj} in {drawer}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj=obj,
                    drawer=drawer,
                    location="drawer-area"
                )
                if done:
                    print(f"[Exploration] Placed {obj} in {drawer} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to place {obj} in {drawer}: {e}")

        # 4. Try to push drawers closed
        for drawer in drawer_list:
            try:
                print(f"[Exploration] Attempting to push {drawer}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer,
                    location="drawer-area"
                )
                if done:
                    print(f"[Exploration] Pushed {drawer} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to push {drawer}: {e}")

        # 5. Try to move to all locations
        for loc in location_list:
            try:
                print(f"[Exploration] Attempting to move to {loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location="ready-pose",
                    to_location=loc
                )
                if done:
                    print(f"[Exploration] Moved to {loc} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # 6. Try to sweep and rotate (if applicable)
        for obj in object_list:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at floor")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj=obj,
                    location="floor"
                )
                if done:
                    print(f"[Exploration] Swept {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

            try:
                print(f"[Exploration] Attempting to rotate {obj} at floor")
                obs, reward, done = execute_rotate(
                    env,
                    task,
                    obj=obj,
                    location="floor"
                )
                if done:
                    print(f"[Exploration] Rotated {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to rotate {obj}: {e}")

        # 7. Try to use gripper (if applicable)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Used gripper successfully.")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate errors.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
