# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List (Cleaned as per feedback) ===
        # Only include objects that are present in the environment/code
        object_names = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2"
        ]
        # Comment: Each object name corresponds to a physical object in the environment.
        # Handles are for drawers, places are for placing objects, dice are manipulable objects.

        # === Safety/Existence Check Helper ===
        def is_object_present(obj_name):
            return obj_name in positions

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment to determine which predicate is missing.
        # We will use the available skills to interact with objects and check for missing information.

        # For demonstration, let's try to identify all objects and check for missing predicates.
        # We'll use execute_sweep (as a generic exploration), execute_go (move), and execute_gripper (if needed).
        # We'll also print debug statements to verify object presence.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Assume the robot starts at a known location, e.g., "ready-pose"
        current_location = "ready-pose"
        explored_objects = set()

        # Step 1: Move to each object location and attempt to identify it
        for obj_name in object_names:
            if not is_object_present(obj_name):
                print(f"[Warning] Object '{obj_name}' not present in environment positions. Skipping.")
                continue

            obj_pos = positions[obj_name]
            print(f"[Exploration] Moving to {obj_name} at position {obj_pos}.")

            try:
                # Move to the object's location
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location,
                    to_location=obj_name,  # Use object name as location key
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                current_location = obj_name

                # Attempt to identify the object (simulate exploration)
                print(f"[Exploration] Sweeping/identifying object '{obj_name}'.")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    target_object=obj_name,
                    location=obj_name,
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )
                explored_objects.add(obj_name)

            except Exception as e:
                print(f"[Error] Exception during exploration of '{obj_name}': {e}")

        # Step 2: Attempt to pick and place dice objects to further explore predicates
        for dice_name in ["dice1", "dice2"]:
            if not is_object_present(dice_name):
                print(f"[Warning] Dice object '{dice_name}' not present. Skipping pick/place.")
                continue

            dice_pos = positions[dice_name]
            try:
                # Move to dice location if not already there
                if current_location != dice_name:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_location,
                        to_location=dice_name,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    current_location = dice_name

                # Pick the dice
                print(f"[Exploration] Attempting to pick '{dice_name}'.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=dice_name,
                    location=dice_name,
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )

                # Place the dice at a known place location
                place_location = "drawer_middle_place"
                if is_object_present(place_location):
                    print(f"[Exploration] Placing '{dice_name}' at '{place_location}'.")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        target_object=dice_name,
                        drawer="drawer_middle_handle",  # Example drawer
                        location=place_location,
                        max_steps=50,
                        threshold=0.01,
                        timeout=5.0
                    )
                    current_location = place_location
                else:
                    print(f"[Warning] Place location '{place_location}' not present. Skipping place.")

            except Exception as e:
                print(f"[Error] Exception during pick/place of '{dice_name}': {e}")

        # Step 3: Attempt to pull and push drawers to check for lock-known predicate
        for drawer_handle in ["drawer_top_handle", "drawer_middle_handle", "drawer_bottom_handle"]:
            if not is_object_present(drawer_handle):
                print(f"[Warning] Drawer handle '{drawer_handle}' not present. Skipping pull/push.")
                continue

            try:
                # Move to drawer handle location if not already there
                if current_location != drawer_handle:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_location,
                        to_location=drawer_handle,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    current_location = drawer_handle

                # Pick the handle
                print(f"[Exploration] Attempting to pick handle '{drawer_handle}'.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=drawer_handle,
                    location=drawer_handle,
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )

                # Pull the drawer (simulate lock-known exploration)
                print(f"[Exploration] Attempting to pull '{drawer_handle}'.")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer_handle.replace("_handle", ""),  # Example: "drawer_top"
                    handle=drawer_handle,
                    location=drawer_handle,
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )

                # Push the drawer back (if possible)
                print(f"[Exploration] Attempting to push '{drawer_handle}'.")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer_handle.replace("_handle", ""),
                    location=drawer_handle,
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )

            except Exception as e:
                print(f"[Error] Exception during pull/push of '{drawer_handle}': {e}")

        # Step 4: Use gripper skill as a generic check
        try:
            print("[Exploration] Executing gripper skill for calibration/check.")
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=20,
                timeout=2.0
            )
        except Exception as e:
            print(f"[Error] Exception during gripper skill: {e}")

        print("[Exploration] Exploration phase complete. Checked for missing predicates and object presence.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
