# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests that object placement in the drawer is not robustly checked.
        # We need to explore which predicate is missing for correct placement.
        # We'll use available skills to explore the environment and check for missing predicates.

        # List of objects and drawers (from the improved code example and PDDL)
        dice_list = ['dice1', 'dice2']
        handle_list = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_list = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_list = ['floor', 'drawer-area', 'ready-pose']

        # For exploration, we will try to pick and place dice into drawers and observe if any predicate is missing.
        # We will also check if the object is properly placed in the drawer after the action.

        # For each die, try to pick and place into an available drawer
        for die in dice_list:
            # 1. Move to the die's location (assume on floor)
            try:
                die_pos = positions.get(die, None)
                if die_pos is None:
                    print(f"[Exploration] Position for {die} not found, skipping.")
                    continue
                # Move robot to die's location (assume 'floor')
                obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location='floor')
            except Exception as e:
                print(f"[Exploration] Error moving to {die}: {e}")
                continue

            # 2. Pick the die
            try:
                obs, reward, done = execute_pick(env, task, object_name=die, location='floor')
            except Exception as e:
                print(f"[Exploration] Error picking {die}: {e}")
                continue

            # 3. For each drawer, try to place the die
            for drawer in drawer_list:
                try:
                    # Move to drawer area if needed
                    obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer-area')
                except Exception as e:
                    print(f"[Exploration] Error moving to drawer-area for {drawer}: {e}")
                    continue

                # Try to open the drawer if not open
                try:
                    # Find the handle for this drawer
                    handle = None
                    for h in handle_list:
                        if (h.startswith('handle') and drawer in h):
                            handle = h
                            break
                    if handle is None:
                        # Fallback: use mapping from improved code
                        if drawer == 'drawer_bottom':
                            handle = 'handle_bottom'
                        elif drawer == 'drawer_middle':
                            handle = 'handle_middle'
                        elif drawer == 'drawer_top':
                            handle = 'handle_top'
                    # Pick the handle if not already holding
                    obs, reward, done = execute_pick(env, task, object_name=handle, location='drawer-area')
                    # Pull the drawer open
                    obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location='drawer-area')
                except Exception as e:
                    print(f"[Exploration] Error opening {drawer}: {e}")
                    continue

                # Try to place the die in the drawer
                try:
                    obs, reward, done = execute_place(env, task, object_name=die, drawer=drawer, location='drawer-area')
                except Exception as e:
                    print(f"[Exploration] Error placing {die} in {drawer}: {e}")
                    continue

                # Check if the die is now in the drawer (simulate predicate check)
                # This is where we would check for the missing predicate
                # For now, print the result and continue
                print(f"[Exploration] Placed {die} in {drawer}. Checking placement...")

                # (Optional) Add more robust checks here if available in the environment
                # For example, check if the die's position is within the drawer's bounds
                # or if a predicate like (in-drawer die drawer) is true in the environment

                # After placement, push the drawer closed
                try:
                    obs, reward, done = execute_push(env, task, drawer=drawer, location='drawer-area')
                except Exception as e:
                    print(f"[Exploration] Error closing {drawer}: {e}")
                    continue

                # Return to ready pose after each attempt
                try:
                    obs, reward, done = execute_go(env, task, from_location='drawer-area', to_location='ready-pose')
                except Exception as e:
                    print(f"[Exploration] Error returning to ready-pose: {e}")
                    continue

        print("[Exploration] Completed. If object placement is not robust, a missing predicate (e.g., object alignment or in-drawer check) may be required.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
