# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List for Safety Check ===
        object_names = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2",
            "trash",
            "trash_bin"
        ]

        def is_object_in_list(object_name):
            return object_name in object_names

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests we need to ensure the object is in the object list before manipulation.
        # We also need to explore the environment to check for missing predicates (e.g., identified, weight-known, etc.)
        # We'll use the available exploration actions to do this.

        # For demonstration, let's try to identify all objects at each location
        # and check if any object is missing the 'identified' predicate.

        # Assume we have a list of locations from positions
        location_names = set()
        for obj, pos in positions.items():
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, str):
                location_names.add(pos)
        location_names = list(location_names)
        if not location_names:
            # Fallback: use some default locations if not provided
            location_names = [
                "drawer_top_place_left",
                "drawer_top_place_right",
                "drawer_middle_place",
                "drawer_bottom_place_left",
                "drawer_bottom_place_right",
                "floor",
                "trash_bin"
            ]

        # Assume robot starts at 'ready-pose' or first location
        robot_location = "ready-pose" if "ready-pose" in location_names else location_names[0]
        robot_name = "robot"  # Placeholder, not used in skills

        # Exploration: Go to each location and identify objects
        for loc in location_names:
            if loc == robot_location:
                continue
            try:
                # Use execute_go to move to the location
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=loc)
                robot_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

            # Use execute_sweep as a placeholder for exploration/identification
            # (since execute_go_identify is not in the available skills)
            for obj in object_names:
                if obj in positions and (positions[obj] == loc or (isinstance(positions[obj], dict) and positions[obj].get('location') == loc)):
                    try:
                        obs, reward, done, info = execute_sweep(env, task, obj, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to sweep/identify {obj} at {loc}: {e}")

        # === Main Task Plan ===
        # Example: Insert a die into a drawer, ensuring object is in the list

        # Select a die to insert
        die_candidates = ["dice1", "dice2"]
        die_to_insert = None
        for die in die_candidates:
            if die in positions and is_object_in_list(die):
                die_to_insert = die
                break

        if die_to_insert is None:
            print("[Task] No die found in object list and positions. Aborting task.")
            return

        # Choose a drawer and handle
        drawer = "drawer_middle_place"
        drawer_handle = "drawer_middle_handle"
        drawer_location = drawer  # Assume location name matches object name

        # 1. Move to die location
        die_location = positions[die_to_insert] if isinstance(positions[die_to_insert], str) else positions[die_to_insert].get('location', None)
        if die_location is None:
            print(f"[Task] Could not determine location for {die_to_insert}. Aborting.")
            return

        if robot_location != die_location:
            try:
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=die_location)
                robot_location = die_location
            except Exception as e:
                print(f"[Task] Failed to move to die location {die_location}: {e}")
                return

        # 2. Pick up the die
        try:
            obs, reward, done, info = execute_pick(env, task, die_to_insert, die_location)
        except Exception as e:
            print(f"[Task] Failed to pick up {die_to_insert} at {die_location}: {e}")
            return

        # 3. Return to ready-pose if required (simulate need-ready)
        try:
            obs, reward, done, info = execute_go_ready(env, task, from_location=die_location)
            robot_location = "ready-pose"
        except Exception as e:
            print(f"[Task] Failed to return to ready-pose: {e}")

        # 4. Move to drawer handle location
        handle_location = drawer_handle  # Assume handle is at its own location
        if robot_location != handle_location:
            try:
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=handle_location)
                robot_location = handle_location
            except Exception as e:
                print(f"[Task] Failed to move to handle location {handle_location}: {e}")
                return

        # 5. Pick up the handle
        try:
            obs, reward, done, info = execute_pick(env, task, drawer_handle, handle_location)
        except Exception as e:
            print(f"[Task] Failed to pick up handle {drawer_handle} at {handle_location}: {e}")
            return

        # 6. Pull the drawer open
        try:
            obs, reward, done, info = execute_pull(env, task, drawer, drawer_handle, handle_location)
        except Exception as e:
            print(f"[Task] Failed to pull open drawer {drawer} with handle {drawer_handle}: {e}")
            return

        # 7. Move to drawer location (if not already there)
        if robot_location != drawer_location:
            try:
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=drawer_location)
                robot_location = drawer_location
            except Exception as e:
                print(f"[Task] Failed to move to drawer location {drawer_location}: {e}")
                return

        # 8. Place the die into the drawer
        try:
            obs, reward, done, info = execute_place(env, task, die_to_insert, drawer, drawer_location)
        except Exception as e:
            print(f"[Task] Failed to place {die_to_insert} into {drawer}: {e}")
            return

        # 9. Return to ready-pose if required (simulate need-ready)
        try:
            obs, reward, done, info = execute_go_ready(env, task, from_location=drawer_location)
            robot_location = "ready-pose"
        except Exception as e:
            print(f"[Task] Failed to return to ready-pose after placing die: {e}")

        # 10. Push the drawer closed
        try:
            obs, reward, done, info = execute_push(env, task, drawer, drawer_location)
        except Exception as e:
            print(f"[Task] Failed to push drawer {drawer} closed: {e}")

        print("[Task] Successfully inserted die into drawer and closed it.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
