# run_skeleton_task.py (Executable with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills, do not redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase to Identify Missing Predicate ===
        print("[Exploration] Starting to identify missing predicates needed in planning...")

        # Approach:
        # - For each object and location, use available skills and observe state changes.
        # - Log outcomes to help identify which facts/predicates are NOT captured and needed.

        # Assumption: positions dict includes 'robot', drawers, handles, and objects.

        # Example naming conventions used in positions:
        #   - 'robot': pos
        #   - 'drawer1': pos
        #   - 'object1': pos
        #   - 'handle_drawer1': pos
        # Find key names and location targets
        robot_key = None
        for k in positions.keys():
            if "robot" in k:
                robot_key = k
                break
        if robot_key is None:
            robot_key = list(positions.keys())[0]  # Fallback

        object_keys = [k for k in positions.keys() if "object" in k]
        handle_keys = [k for k in positions.keys() if "handle" in k or "Handle" in k]
        drawer_keys = [k for k in positions.keys() if "drawer" in k or "Drawer" in k]

        # Use first available object, handle, drawer for exploration if present
        explore_object = object_keys[0] if object_keys else None
        explore_handle = handle_keys[0] if handle_keys else None
        explore_drawer = drawer_keys[0] if drawer_keys else None

        # --- Try various primitive skills, observe what fails ---
        exploration_steps = []
        # 1. Try to execute_pick on an object (should require on-floor, hand-empty, etc.)
        if explore_object:
            print(f"[Exploration] Attempting to execute_pick on {explore_object}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target=explore_object,
                    location=positions[explore_object],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_steps.append('execute_pick')
                if done:
                    print(f"[Exploration] Successfully executed pick on {explore_object}")
                else:
                    print(f"[Exploration] Pick not completed, probably due to a missing predicate")
            except Exception as e:
                print(f"[Exploration] execute_pick on {explore_object} failed: {e}")

        # 2. Try to execute_place if possible (must have picked up an object)
        if explore_object and explore_drawer:
            print(f"[Exploration] Attempting to execute_place {explore_object} into {explore_drawer}")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    target=explore_object,
                    drawer=explore_drawer,
                    location=positions[explore_drawer],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_steps.append('execute_place')
                if done:
                    print(f"[Exploration] Successfully executed place {explore_object} in {explore_drawer}")
                else:
                    print(f"[Exploration] Place not completed, possibly indicates missing predicate such as drawer status")
            except Exception as e:
                print(f"[Exploration] execute_place encountered an error: {e}")

        # 3. Try to execute_pull on a handle if available
        if explore_drawer and explore_handle:
            print(f"[Exploration] Attempting to execute_pull {explore_drawer} using handle {explore_handle}")
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=explore_drawer,
                    handle=explore_handle,
                    location=positions[explore_drawer],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_steps.append('execute_pull')
                if done:
                    print(f"[Exploration] Pull succeeded")
                else:
                    print(f"[Exploration] Pull did not succeed, possibly due to a missing lock/unlock predicate")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try to execute_go (move) between two locations if possible
        available_locs = list(positions.keys())
        if len(available_locs) >= 2:
            loc1, loc2 = available_locs[0], available_locs[1]
            print(f"[Exploration] Attempting execute_go from {loc1} to {loc2}")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[loc1],
                    to_location=positions[loc2],
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_steps.append('execute_go')
                if done:
                    print(f"[Exploration] Robot moved from {loc1} to {loc2}")
                else:
                    print(f"[Exploration] Movement not completed, possibly a constraint (need-ready/robot-free) missing")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 5. Try to execute_push on the drawer
        if explore_drawer:
            print(f"[Exploration] Attempting to execute_push {explore_drawer}")
            try:
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=explore_drawer,
                    location=positions[explore_drawer],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_steps.append('execute_push')
                if done:
                    print(f"[Exploration] Drawer pushed closed.")
                else:
                    print(f"[Exploration] Drawer could not be closed, might be due to missing preconditions such as drawer-open?")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to execute_go_ready (if available), having set up a state requiring need-ready
        # Not all domains support go_ready directly, but try if implemented
        if hasattr(task, "robot_location"):  # just an example check
            print(f"[Exploration] Attempting to execute_go_ready to ready-pose")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=positions[robot_key],
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_steps.append('execute_go_ready')
                if done:
                    print(f"[Exploration] Robot returned to ready-pose.")
                else:
                    print(f"[Exploration] Could not execute_go_ready, may require need-ready flag")
            except Exception as e:
                print(f"[Exploration] execute_go_ready failed: {e}")

        print(f"[Exploration] Completed primitive action attempts: {exploration_steps}")
        print("[Exploration] Please review above outputs for indications of missing predicates in preconditions/effects.")

        # === END OF EXPLORATION PHASE ===

        # You can insert further logic for running the actual oracle plan here,
        # now informed by the predicate(s) discovered during exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
