# run_skeleton_task.py (Completed with Exploration and Object Validation)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def validate_object(obj_name, object_names):
    """Check if the object name exists in the list of object names."""
    return obj_name in object_names

def get_object(obj_name, object_list):
    """Retrieve the object dict from the object list by name."""
    for obj in object_list:
        if obj['name'] == obj_name:
            return obj
    return None

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Object List ===
        # positions: dict of object_name -> position
        positions = get_object_positions()
        # For validation, get the list of object names
        object_names = list(positions.keys())
        # If you have a richer object_list (e.g., with types), you can retrieve it here
        # For now, we assume only names are available

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for missing predicates
        # such as 'identified', 'temperature-known', 'weight-known', 'durability-known', 'lock-known'
        # For this example, we will attempt to identify all objects at each location

        # Suppose we want to ensure all objects are 'identified' before manipulation
        # We'll simulate this by iterating over all objects and performing an exploration skill if needed

        # For demonstration, let's assume the robot starts at a known location
        # and we want to identify all objects at their locations

        # Let's get the robot's current location from the observation if possible
        robot_location = None
        if 'robot' in positions:
            robot_location = positions['robot']
        else:
            # Fallback: try to find a key that looks like the robot
            for k in positions:
                if 'robot' in k or 'base' in k:
                    robot_location = positions[k]
                    break

        # For this example, we assume locations are the set of unique positions of all objects
        location_set = set()
        for obj_name, pos in positions.items():
            if obj_name != 'robot':
                location_set.add(tuple(pos))
        location_list = list(location_set)

        # For each location, move the robot there and perform an identification skill
        for loc in location_list:
            # Find all objects at this location
            objs_at_loc = [name for name, pos in positions.items() if tuple(pos) == loc and name != 'robot']
            if not objs_at_loc:
                continue
            # Move robot to this location if not already there
            # (Assume execute_go takes from_location and to_location as arguments)
            # For this example, we use the object name as a proxy for location
            # In a real scenario, you would map positions to location names
            target_obj = objs_at_loc[0]
            if not validate_object(target_obj, object_names):
                print(f"[Exploration] Object {target_obj} not found in object list, skipping.")
                continue
            try:
                # Use execute_go to move to the object's location
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=positions[target_obj],
                    max_steps=100
                )
                robot_location = positions[target_obj]
            except Exception as e:
                print(f"[Exploration] Failed to move to {target_obj}: {e}")
                continue
            try:
                # Use execute_sweep or another exploration skill to identify the object
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    target_obj,
                    robot_location,
                    max_steps=50
                )
                print(f"[Exploration] Identified object {target_obj} at location {robot_location}")
            except Exception as e:
                print(f"[Exploration] Failed to identify {target_obj}: {e}")

        # === Main Task Plan ===
        # Example: Pick up an object and place it in a drawer
        # For demonstration, let's pick the first object and place it if possible

        # Find a non-handle object on the floor
        pick_obj = None
        for obj_name in object_names:
            if 'handle' not in obj_name and 'drawer' not in obj_name and obj_name != 'robot':
                pick_obj = obj_name
                break

        if pick_obj is not None and validate_object(pick_obj, object_names):
            try:
                # Pick the object
                obs, reward, done = execute_pick(
                    env,
                    task,
                    pick_obj,
                    positions[pick_obj],
                    max_steps=100
                )
                print(f"[Task] Picked up object {pick_obj}")
            except Exception as e:
                print(f"[Task] Failed to pick up {pick_obj}: {e}")
        else:
            print("[Task] No valid object found to pick.")

        # Example: Place the object in a drawer if a drawer is available and open
        drawer_obj = None
        for obj_name in object_names:
            if 'drawer' in obj_name:
                drawer_obj = obj_name
                break

        if pick_obj and drawer_obj and validate_object(drawer_obj, object_names):
            try:
                # Place the object in the drawer
                obs, reward, done = execute_place(
                    env,
                    task,
                    pick_obj,
                    drawer_obj,
                    positions[drawer_obj],
                    max_steps=100
                )
                print(f"[Task] Placed object {pick_obj} in {drawer_obj}")
            except Exception as e:
                print(f"[Task] Failed to place {pick_obj} in {drawer_obj}: {e}")
        else:
            print("[Task] No valid drawer found for placing.")

        # === End of Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
