# run_skeleton_task.py (Task with Exploration to Find Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Run the robotic task and explicitly explore to find missing predicates as suggested by feedback.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions: dict of object names -> (x, y, z) or as required by your skill functions

        # ==============================
        # EXPLORATION PHASE - Find missing predicate using exploration knowledge 
        # ==============================

        # As indicated by the feedback: "(ready-pose)" is a missing predicate.
        # The goal here is to explore the environment and generate information about states that might be missing,
        # particularly the (ready-pose) predicate or its effect.
        # We assume that to check for the existence or necessity of (ready-pose), we need to:
        # - Move the robot to the 'ready-pose' location (if it exists).
        # - Try interacting with the relevant skills.
        # Since we do not have skill functions for 'execute_go_identify', etc., 
        # and can only use ones listed in skill_code, 
        # we will attempt to move to a location named 'ready-pose' if it exists, and perform 'execute_go_ready' if available.

        # Determine whether "ready-pose" is a known location in the current environment
        known_locations = []
        for key, value in positions.items():
            if 'pose' in key or 'ready' in key:
                known_locations.append(key)
        if 'ready-pose' not in known_locations:
            known_locations.append('ready-pose')  # Fallback in case not listed in positions

        # Find current robot location (assume env/task provides this in obs or positions)
        robot_location = None
        for key, pos in positions.items():
            if 'robot' in key or 'Robot' in key:
                robot_location = key
                break

        # If no robot location found from positions, try to infer
        if robot_location is None:
            robot_location = 'init'

        # Try exploration step: Move robot to ready-pose (if such a key exists)
        print("[Exploration] Attempting to move robot to 'ready-pose' to check for missing (ready-pose) predicate.")

        # The 'execute_go_ready' action is available in the PDDL as an action/move to ready-pose.
        # But it is not in the provided skill list. So, use 'execute_go' as a general move,
        # then attempt 'execute_go_ready' if it exists in skill_code.

        # 1. Move to ready-pose using 'execute_go'
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location='ready-pose'  # Using the name from domain PDDL
            )
            print("[Exploration] Robot moved to 'ready-pose'.")
        except Exception as e:
            print(f"[Exploration] Could not move to 'ready-pose' with execute_go: {e}")

        # 2. Try calling 'execute_go_ready' if it is a skill (uncertain, as it's not in the skill names list)
        # If not, fallback: Take an additional step in ready-pose to trigger any effect
        if 'execute_go_ready' in globals():
            try:
                obs, reward, done = execute_go_ready(env, task, from_location='ready-pose')
                print("[Exploration] Executed execute_go_ready to ensure (ready-pose) is registered.")
            except Exception as e:
                print(f"[Exploration] execute_go_ready failed: {e}")

        # 3. Optionally: Try using other skills at 'ready-pose' to see if (need-ready) predicate is affected.
        # (Here, just simulate a normal interaction as a placeholder exploration.)
        # For example, try gripper action at ready-pose.
        try:
            if 'execute_gripper' in globals():
                obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Gripper action at ready-pose failed: {e}")

        print("[Exploration] Exploration for missing (ready-pose) predicate complete.")

        # ==============================
        # Here, continue with any oracle plan as needed
        # (You may insert the oracle plan steps here using only available skills.)
        # ==============================

        # The rest of your planned actions/task logic would follow here.
        # For this exploration cycle, the primary goal is to discover/state the missing (ready-pose).
        print("[Task] Oracle plan would be executed after exploration, using only available skills... (Not shown)")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()