# run_skeleton_task.py (Filled in with object validation and exploration as per instructions)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use the available predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording if needed
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all object positions in the scene
        positions = get_object_positions()

        # Try to determine available object names/types from env/task/positions
        # Assume a function get_all_object_names() or infer from positions keys
        try:
            object_names = list(positions.keys())
        except Exception as e:
            print("[ERROR] Could not retrieve object names from positions:", e)
            object_names = []

        # Assemble a set for fast validation
        object_set = set(object_names)

        # Example goal: "Pick and place an object into a drawer"
        # For demonstration, try to pick an object and explore missing predicates
        # Names below are exemplary; adapt as per your environment

        # User must specify/select one object and one drawer for the generic task
        # We'll try to select the first object not a handle and a drawer
        selected_object = None
        selected_drawer = None
        selected_handle = None
        for name in object_names:
            lname = name.lower()
            if "drawer" in lname and "handle" not in lname and selected_drawer is None:
                selected_drawer = name
            elif "handle" in lname and selected_handle is None:
                selected_handle = name
            elif selected_object is None and "drawer" not in lname and "handle" not in lname:
                selected_object = name

        # --- Object presence/type safety checks ---
        # Validate selected objects from object list
        if selected_object is None:
            print("[ERROR] No suitable pickup object found in object list:", object_names)
        if selected_drawer is None:
            print("[ERROR] No suitable drawer found in object list:", object_names)
        if selected_handle is None:
            print("[ERROR] No suitable handle found in object list:", object_names)

        # Further logic/validation: check basic types or any available meta info
        # For demonstration, if any object is missing, abort safely
        if selected_object not in object_set or selected_drawer not in object_set or selected_handle not in object_set:
            print("[ERROR] One or more required objects are not present. Aborting.")
            return

        print(f"[INFO] Using object: '{selected_object}', drawer: '{selected_drawer}', handle: '{selected_handle}'")

        # Gather possible locations from positions (choose the first unique for each object)
        object_loc = positions[selected_object] if selected_object in positions else None
        drawer_loc = positions[selected_drawer] if selected_drawer in positions else None
        handle_loc = positions[selected_handle] if selected_handle in positions else None

        # This assumes location names (e.g., position dicts) are compatible with skill_code arguments.

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # As per feedback and the domain, "need-ready" is a possible missing predicate
        # We'll simulate exploration: robot goes to the object's location, identifies and checks predicates

        print("[Exploration] Determining missing predicate via exploration phase...")
        # Move to the target object's location to 'identify' it
        try:
            execute_go(env, task, from_location="ready-pose", to_location=object_loc)
            # Simulate identification (predicate: identified)
            print(f"[Exploration] Arrived at {object_loc} to identify {selected_object}")
        except Exception as e:
            print(f"[Exploration] Failed to go to {object_loc}: {e}")
            return

        # Check temperature (simulate as another exploration predicate)
        try:
            execute_go(env, task, from_location=object_loc, to_location=object_loc)  # Stay at same loc for exploration
            print(f"[Exploration] Temperature checked for {selected_object}")
        except Exception as e:
            print(f"[Exploration] Failed temperature exploration: {e}")
            return

        # Try picking up the object to check its weight (simulate force calibration)
        try:
            result = execute_pick(env, task, obj_name=selected_object, location=object_loc)
            print(f"[Exploration] Weight checked for {selected_object}")
        except Exception as e:
            print(f"[Exploration] Failed weight check / execute_pick: {e}")
            return

        # Try to pick up the drawer handle next (simulate durability/lock exploration)
        try:
            result = execute_pick(env, task, obj_name=selected_handle, location=handle_loc)
            print(f"[Exploration] Durability checked / Handle picked: {selected_handle}")
        except Exception as e:
            print(f"[Exploration] Failed handle pick: {e}")
            return

        # Try to pull the drawer to explore lock state
        try:
            result = execute_pull(env, task, drawer_name=selected_drawer, handle_name=selected_handle, location=drawer_loc)
            print(f"[Exploration] Lock state explored by pulling {selected_drawer}")
        except Exception as e:
            print(f"[Exploration] Failed at pull/exploring lock: {e}")
            return

        # After exploration, output what was "learned" (In real task: update a knowledge base or print results)
        print("[Exploration] Exploration phase complete. Missing predicate (need-ready) was explored.")

        # === Main Oracle Plan: Demonstrate Picking & Placing with Predicate Validation ===

        print("[Task] Starting main plan: pick object, go ready-pose, place in drawer, close drawer.")

        # Pick the object (ensure all preconditions incl. not (need-ready) are met)
        try:
            result = execute_pick(env, task, obj_name=selected_object, location=object_loc)
            print(f"[Task] Picked object: {selected_object}")
        except Exception as e:
            print(f"[Task] Could not pick {selected_object}: {e}")
            return

        # After pick, 'need-ready' is typically set; must execute_go_ready before any further actions
        try:
            result = execute_go_ready(env, task, from_location=object_loc)
            print(f"[Task] Returned to ready-pose after picking {selected_object}.")
        except Exception as e:
            print(f"[Task] Could not go to ready-pose: {e}")
            return

        # Move to drawer's location
        try:
            result = execute_go(env, task, from_location="ready-pose", to_location=drawer_loc)
            print(f"[Task] Arrived at drawer location: {drawer_loc}")
        except Exception as e:
            print(f"[Task] Could not move to drawer: {e}")
            return

        # Place the object into the drawer
        try:
            result = execute_place(env, task, obj_name=selected_object, drawer_name=selected_drawer, location=drawer_loc)
            print(f"[Task] Placed object: {selected_object} into drawer: {selected_drawer}")
        except Exception as e:
            print(f"[Task] Could not place: {e}")
            return

        # After place, 'need-ready' is set; must return to ready pose
        try:
            result = execute_go_ready(env, task, from_location=drawer_loc)
            print(f"[Task] Returned to ready-pose after placing object.")
        except Exception as e:
            print(f"[Task] Could not go to ready-pose: {e}")
            return

        # Optionally close the drawer (push)
        try:
            result = execute_push(env, task, drawer_name=selected_drawer, location=drawer_loc)
            print(f"[Task] Drawer closed: {selected_drawer}")
        except Exception as e:
            print(f"[Task] Could not push/close drawer: {e}")

        print("===== Task Complete =====")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
