# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names as per the PDDL problem ===
        object_names = [
            'dice1', 'dice2',
            'handle_bottom', 'handle_middle', 'handle_top'
        ]
        drawer_names = [
            'drawer_bottom', 'drawer_middle', 'drawer_top'
        ]
        location_names = [
            'floor', 'drawer-area', 'ready-pose'
        ]

        # === Helper: Check if object exists in positions ===
        def object_exists(obj_name):
            return obj_name in positions

        # === Exploration Phase: Identify missing predicate ===
        # The feedback suggests that sometimes an object (e.g., dice1) is not in the object list.
        # We need to check for this before attempting to manipulate it.
        # If not found, log and skip; otherwise, proceed.

        # For demonstration, let's try to pick and place dice1 and dice2 into drawer_bottom.
        # We will also ensure the robot is at the correct location and the drawer is open.

        # --- Step 1: Move to ready-pose if not already there ---
        try:
            robot_pos = None
            for loc in location_names:
                if 'robot-at' in descriptions and loc in descriptions['robot-at']:
                    robot_pos = loc
                    break
            if robot_pos != 'ready-pose':
                # Move to ready-pose
                obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location='ready-pose')
        except Exception as e:
            print("[Exploration] Could not determine robot position, skipping move to ready-pose.", e)

        # --- Step 2: For each dice, check existence and try to pick and place ---
        for dice in ['dice1', 'dice2']:
            if not object_exists(dice):
                print(f"[Exploration] Error: '{dice}' not found in object list. Skipping.")
                continue

            # Move to floor if not already there
            try:
                obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location='floor')
            except Exception as e:
                print(f"[Exploration] Could not move to floor for {dice}: {e}")
                continue

            # Try to pick the dice
            try:
                obs, reward, done = execute_pick(env, task, object_name=dice, location='floor')
            except Exception as e:
                print(f"[Exploration] Could not pick {dice}: {e}")
                continue

            # After picking, move to drawer-area
            try:
                obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer-area')
            except Exception as e:
                print(f"[Exploration] Could not move to drawer-area with {dice}: {e}")
                continue

            # Open the drawer if needed (pull handle)
            # First, check if the drawer is closed and unlocked
            try:
                # Try to pick the handle if not already holding
                handle = 'handle_bottom'
                if object_exists(handle):
                    obs, reward, done = execute_pick_handle(env, task, object_name=handle, location='drawer-area')
                    # Pull the drawer
                    obs, reward, done = execute_pull(env, task, drawer_name='drawer_bottom', handle_name=handle, location='drawer-area')
                else:
                    print(f"[Exploration] Error: '{handle}' not found in object list. Skipping drawer opening.")
            except Exception as e:
                print(f"[Exploration] Could not open drawer_bottom: {e}")

            # Place the dice in the drawer
            try:
                obs, reward, done = execute_place(env, task, object_name=dice, drawer_name='drawer_bottom', location='drawer-area')
            except Exception as e:
                print(f"[Exploration] Could not place {dice} in drawer_bottom: {e}")

            # After placing, return to ready-pose (if needed)
            try:
                obs, reward, done = execute_go_ready(env, task, from_location='drawer-area')
            except Exception as e:
                print(f"[Exploration] Could not return to ready-pose after placing {dice}: {e}")

        # --- Step 3: Exploration for missing predicates ---
        # If any object is missing, log it and suggest that the missing predicate is likely (on-floor ?o) or (at ?o ?loc)
        for obj in object_names:
            if not object_exists(obj):
                print(f"[Exploration] Missing predicate for object: {obj}. Likely missing (on-floor {obj}) or (at {obj} <location>) in initial state.")

        # --- Step 4: End of task ---
        print("[Task] Exploration and main plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
