# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Discovery ===
        # Try to get all objects and drawers from the environment descriptions or positions
        # Fallback to some default names if not found
        object_names = []
        drawer_names = []
        location_names = []
        try:
            # Try to extract object and drawer names from positions
            for name in positions.keys():
                if "dice" in name or "object" in name:
                    object_names.append(name)
                if "drawer" in name:
                    drawer_names.append(name)
                if "location" in name or "pose" in name or "table" in name:
                    location_names.append(name)
        except Exception as e:
            print("[Warning] Could not extract object/drawer/location names from positions:", e)

        # Fallbacks if not found
        if not object_names:
            object_names = ["dice1", "dice2"]
        if not drawer_names:
            drawer_names = ["drawer1"]
        if not location_names:
            location_names = ["table", "ready-pose"]

        # === Safety and Presence Checks ===
        # Check that at least one dice is present in the environment
        dice_to_place = None
        for dice in ["dice1", "dice2"]:
            if dice in object_names:
                dice_to_place = dice
                break
        if dice_to_place is None:
            print("[Error] No dice object found in the environment. Aborting task.")
            return

        # Check that a drawer is present
        drawer = drawer_names[0] if drawer_names else None
        if drawer is None:
            print("[Error] No drawer found in the environment. Aborting task.")
            return

        # === Determine Initial Robot Location ===
        robot_location = None
        try:
            # Try to get robot location from observation or positions
            if hasattr(task, "get_robot_location"):
                robot_location = task.get_robot_location()
            elif "robot" in positions:
                robot_location = positions["robot"]
            else:
                robot_location = "ready-pose"
        except Exception:
            robot_location = "ready-pose"

        # === Exploration Phase: Discover Missing Predicate (e.g., lock-known) ===
        # The feedback suggests we need to check the drawer's lock state before placing the dice.
        # We use the available skills to explore and discover the lock state.

        # 1. Move to the drawer location (if not already there)
        #    Assume the drawer is at a known location, or use its position
        drawer_location = None
        for loc in positions:
            if drawer in loc:
                drawer_location = loc
                break
        if not drawer_location:
            # Fallback: use the first location or "table"
            drawer_location = location_names[0] if location_names else "table"

        # If robot is not at the drawer location, move there
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location} to explore drawer.")
                obs, reward, done, info = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Warning] Could not move robot to drawer location: {e}")

        # 2. Try to pick the drawer handle (if handle exists)
        #    For exploration, we assume the handle is an object associated with the drawer
        handle_name = None
        for obj in object_names:
            if "handle" in obj and drawer in obj:
                handle_name = obj
                break
        if not handle_name:
            # Fallback: try "handle1" or "drawer1_handle"
            for candidate in ["handle1", f"{drawer}_handle"]:
                if candidate in object_names:
                    handle_name = candidate
                    break

        # If handle is found, try to pick it to enable pull (and thus lock-known)
        picked_handle = False
        if handle_name:
            try:
                print(f"[Exploration] Attempting to pick handle {handle_name} at {drawer_location}.")
                obs, reward, done, info = execute_pick(env, task, handle_name, drawer_location)
                picked_handle = True
            except Exception as e:
                print(f"[Warning] Could not pick handle {handle_name}: {e}")

        # 3. Try to pull the drawer to discover lock state (lock-known)
        #    This is the exploration action to discover the missing predicate
        lock_known = False
        if picked_handle:
            try:
                print(f"[Exploration] Attempting to pull drawer {drawer} using handle {handle_name}.")
                obs, reward, done, info = execute_pull(env, task, drawer, handle_name, drawer_location)
                lock_known = True
                print("[Exploration] Drawer lock state discovered (lock-known).")
            except Exception as e:
                print(f"[Warning] Could not pull drawer {drawer}: {e}")
        else:
            print("[Exploration] Skipping pull: handle not picked.")

        # === Main Task: Place Dice into Drawer ===

        # 4. Move to dice location (if not already there)
        dice_location = None
        for loc in positions:
            if dice_to_place in loc:
                dice_location = loc
                break
        if not dice_location:
            # Fallback: use "table" or first location
            dice_location = "table"

        if robot_location != dice_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {dice_location} to pick dice.")
                obs, reward, done, info = execute_go(env, task, robot_location, dice_location)
                robot_location = dice_location
            except Exception as e:
                print(f"[Warning] Could not move robot to dice location: {e}")

        # 5. Safety check: is dice on the floor and hand empty?
        #    (Assume skill will fail gracefully if not)
        try:
            print(f"[Task] Picking up dice {dice_to_place} at {dice_location}.")
            obs, reward, done, info = execute_pick(env, task, dice_to_place, dice_location)
        except Exception as e:
            print(f"[Warning] Could not pick dice {dice_to_place}: {e}")

        # 6. Move to drawer location (if not already there)
        if robot_location != drawer_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {drawer_location} to place dice.")
                obs, reward, done, info = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
            except Exception as e:
                print(f"[Warning] Could not move robot to drawer location: {e}")

        # 7. Safety check: is drawer open and not full?
        #    (Assume skill will fail gracefully if not)
        try:
            print(f"[Task] Placing dice {dice_to_place} into drawer {drawer} at {drawer_location}.")
            obs, reward, done, info = execute_place(env, task, dice_to_place, drawer, drawer_location)
        except Exception as e:
            print(f"[Warning] Could not place dice {dice_to_place} into drawer {drawer}: {e}")

        # 8. (Optional) Push drawer closed
        try:
            print(f"[Task] Pushing drawer {drawer} closed at {drawer_location}.")
            obs, reward, done, info = execute_push(env, task, drawer, drawer_location)
        except Exception as e:
            print(f"[Warning] Could not push drawer {drawer}: {e}")

        # 9. (Optional) Return to ready pose
        try:
            print(f"[Task] Returning robot to ready-pose.")
            obs, reward, done, info = execute_go(env, task, robot_location, "ready-pose")
            robot_location = "ready-pose"
        except Exception as e:
            print(f"[Warning] Could not return robot to ready-pose: {e}")

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
