# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from feedback and PDDL) ===
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_names = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        drawer_handles = {
            'drawer_bottom': 'handle_bottom',
            'drawer_middle': 'handle_middle',
            'drawer_top': 'handle_top'
        }
        drawer_locations = {
            'drawer_bottom': 'drawer-area',
            'drawer_middle': 'drawer-area',
            'drawer_top': 'drawer-area'
        }
        ready_pose = 'ready-pose'
        floor = 'floor'

        # === Helper: Get current state from task (if available) ===
        # If task exposes a state query, use it; otherwise, rely on positions and assumptions.

        # === Exploration Phase: Find Missing Predicate (e.g., drawer-empty) ===
        # The feedback suggests we must check which drawer is empty before placing dice.
        # We'll explore each drawer and check its state using only available skills.

        # We'll keep track of which drawers are empty.
        empty_drawers = []
        for drawer in drawer_names:
            try:
                # Check if the drawer is empty by using the execute_go skill to move to the drawer-area,
                # then (if available) use execute_sweep or similar to sense contents.
                # Since we don't have a direct sensing skill, we assume the initial state or use positions.
                # If positions[drawer] exists and no dice are in-drawer, we consider it empty.
                # (In a real system, this would be replaced by a perception or query.)
                # For now, we use the initial state from feedback: all drawers are empty at start.
                empty_drawers.append(drawer)
            except Exception as e:
                print(f"[Exploration] Could not determine state of {drawer}: {e}")

        # === Main Task Plan ===
        # For each dice, pick it from the floor and place it into an empty drawer (one per drawer).
        # Only use predefined skills: execute_pick, execute_place, execute_push, execute_pull, execute_go, etc.

        used_drawers = set()
        for dice in dice_names:
            # 1. Check if dice is on the floor (from positions or initial state)
            try:
                # 2. Find an empty drawer not yet used
                target_drawer = None
                for drawer in empty_drawers:
                    if drawer not in used_drawers:
                        target_drawer = drawer
                        used_drawers.add(drawer)
                        break
                if target_drawer is None:
                    print(f"[Task] No empty drawer available for {dice}.")
                    continue

                # 3. Move to the floor (if not already there)
                # Assume robot starts at ready-pose
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=ready_pose,
                    to_location=floor
                )

                # 4. Pick the dice from the floor
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=dice,
                    location=floor
                )

                # 5. Return to ready-pose (if required by need-ready)
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=floor
                )

                # 6. Move to the drawer-area
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=ready_pose,
                    to_location=drawer_locations[target_drawer]
                )

                # 7. Pick the handle for the drawer (if required)
                handle = drawer_handles[target_drawer]
                obs, reward, done = execute_pick_handle(
                    env,
                    task,
                    handle_name=handle,
                    location=drawer_locations[target_drawer]
                )

                # 8. Pull the drawer open
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=target_drawer,
                    handle_name=handle,
                    location=drawer_locations[target_drawer]
                )

                # 9. Place the dice into the drawer
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=dice,
                    drawer_name=target_drawer,
                    location=drawer_locations[target_drawer]
                )

                # 10. Return to ready-pose (if required by need-ready)
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=drawer_locations[target_drawer]
                )

                # 11. Move to the drawer-area again (if needed)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=ready_pose,
                    to_location=drawer_locations[target_drawer]
                )

                # 12. Push the drawer closed
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=target_drawer,
                    location=drawer_locations[target_drawer]
                )

                # 13. Return to ready-pose
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=drawer_locations[target_drawer]
                )

                print(f"[Task] Successfully placed {dice} into {target_drawer}.")

            except Exception as e:
                print(f"[Task] Error handling {dice}: {e}")

        print("[Task] All dice processed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
