# run_skeleton_task.py (Completed with Exploration Phase and Object Validation)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Validate Object List ===
        # Get the actual object names present in the environment
        available_objects = set(positions.keys())
        print("[Info] Available objects in environment:", available_objects)

        # Example: If you have a plan or list of objects to manipulate, validate them
        # For demonstration, let's assume the plan wants to use 'dice1' and 'dice2'
        plan_objects = ['dice1', 'dice2']
        missing_objects = [obj for obj in plan_objects if obj not in available_objects]
        if missing_objects:
            print("[Warning] The following objects are missing from the environment:", missing_objects)
            # Optionally, skip or handle missing objects gracefully
            # For this run, we will not attempt to manipulate missing objects

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to discover a missing predicate
        # We'll perform an exploration step using available skills to probe the environment

        # For demonstration, let's try to identify all objects at each location
        # We'll use the available skill 'execute_go' to move, and simulate an 'identify' phase

        # Get all unique locations from object positions
        all_locations = set()
        for pos in positions.values():
            if isinstance(pos, dict) and 'location' in pos:
                all_locations.add(pos['location'])
            elif isinstance(pos, str):
                all_locations.add(pos)
            elif isinstance(pos, tuple) and len(pos) == 3:
                # If positions are 3D coordinates, skip location-based exploration
                continue

        # If no location info, skip exploration
        if not all_locations:
            print("[Exploration] No location information found in object positions. Skipping exploration phase.")
        else:
            # Assume robot starts at some default location, e.g., 'ready-pose'
            current_location = 'ready-pose'
            for loc in all_locations:
                if loc == current_location:
                    continue
                try:
                    print(f"[Exploration] Moving robot from {current_location} to {loc} to identify objects.")
                    # Use the predefined skill 'execute_go'
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_location,
                        to_location=loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    current_location = loc
                    # After moving, we could attempt to identify objects (if such a skill exists)
                    # Since only predefined skills are allowed, we just log the exploration
                    print(f"[Exploration] Arrived at {loc}. Objects here: {[obj for obj, p in positions.items() if (p == loc or (isinstance(p, dict) and p.get('location') == loc))]}")
                except Exception as e:
                    print(f"[Error] Failed to move to {loc}: {e}")

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan using only predefined skills.
        # For demonstration, let's attempt to pick and place all available objects (excluding missing ones)
        for obj in available_objects:
            try:
                obj_pos = positions[obj]
                # If position is a dict with 'location', extract it
                if isinstance(obj_pos, dict) and 'location' in obj_pos:
                    obj_location = obj_pos['location']
                elif isinstance(obj_pos, str):
                    obj_location = obj_pos
                else:
                    # If position is a 3D coordinate, skip for this logic
                    print(f"[Task] Skipping object {obj} due to unsupported position format.")
                    continue

                print(f"[Task] Attempting to pick object {obj} at location {obj_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj,
                    location=obj_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Successfully picked {obj}.")
                else:
                    print(f"[Task] Failed to pick {obj}.")

                # For demonstration, attempt to place the object at the same location (if place skill exists)
                # You may need to determine a valid target (e.g., a drawer) for place
                # Here, we just log the action
                # obs, reward, done = execute_place(env, task, object_name=obj, location=obj_location, ...)
            except Exception as e:
                print(f"[Error] Exception during pick/place of {obj}: {e}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
