# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from problem definition) ===
        dice_list = ['dice1', 'dice2']
        handle_list = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_list = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_list = ['floor', 'drawer-area', 'ready-pose']

        # === Safety and Force Calibration (Feedback) ===
        # Assume a force calibration skill exists; if not, log and continue
        try:
            if 'execute_gripper' in globals():
                print("[Calibration] Performing force calibration using gripper skill.")
                execute_gripper(env, task)
        except Exception as e:
            print("[Warning] Force calibration step failed or not implemented:", e)

        # === Safety Checks: Ensure all declared objects exist ===
        all_objects = set(positions.keys())
        for obj in dice_list + handle_list + drawer_list:
            if obj not in all_objects:
                print(f"[Error] Declared object '{obj}' not found in environment positions!")
                # Optionally, raise or skip; here, we skip missing objects
                continue

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for a missing predicate,
        # such as lock-known, weight-known, etc. We'll attempt to use available skills to
        # explore and determine which predicate is missing for the drawer or handle.

        # For this example, let's focus on drawer_bottom and handle_bottom
        target_drawer = 'drawer_bottom'
        target_handle = 'handle_bottom'
        target_dice = 'dice1'
        ready_pose = 'ready-pose'
        drawer_area = 'drawer-area'
        floor = 'floor'

        # Step 1: Move to ready-pose if not already there
        try:
            robot_pos = None
            for loc in location_list:
                if positions.get('robot', None) == loc:
                    robot_pos = loc
                    break
            if robot_pos != ready_pose:
                print(f"[Move] Moving robot to {ready_pose}")
                execute_go(env, task, robot_pos if robot_pos else floor, ready_pose)
        except Exception as e:
            print("[Error] Failed to move to ready-pose:", e)

        # Step 2: Exploration - Try to pull the handle to check lock-known predicate
        # (simulate exploration to find missing predicate)
        try:
            print("[Exploration] Attempting to pick handle to check lock status.")
            # Move to handle location if needed
            handle_pos = positions.get(target_handle, drawer_area)
            execute_go(env, task, ready_pose, drawer_area)
            # Pick the handle
            execute_pick(env, task, target_handle, drawer_area)
            # Try to pull the handle (simulate lock-known exploration)
            execute_pull(env, task, target_drawer, target_handle, drawer_area)
            print("[Exploration] Pull action executed to check lock-known predicate.")
        except Exception as e:
            print("[Exploration] Pull action failed (possibly due to lock):", e)
            print("[Exploration] The missing predicate may be lock-known or drawer-locked.")

        # Step 3: Main Task Plan - Insert dice into drawer_bottom
        try:
            # Move to dice location
            print(f"[Task] Moving to pick {target_dice} from {floor}")
            execute_go(env, task, drawer_area, floor)
            # Pick the dice
            execute_pick(env, task, target_dice, floor)
            # Return to ready-pose (if need-ready is set)
            try:
                execute_go_ready(env, task, floor)
            except Exception as e:
                print("[Info] No need to return to ready-pose after pick:", e)
            # Move to drawer area
            print(f"[Task] Moving to drawer area to place {target_dice}")
            execute_go(env, task, ready_pose, drawer_area)
            # Place the dice in the drawer
            execute_place(env, task, target_dice, target_drawer, drawer_area)
            # Return to ready-pose (if need-ready is set)
            try:
                execute_go_ready(env, task, drawer_area)
            except Exception as e:
                print("[Info] No need to return to ready-pose after place:", e)
            print(f"[Task] Successfully placed {target_dice} in {target_drawer}.")
        except Exception as e:
            print("[Error] Failed to complete main task plan:", e)

        # Step 4: Safety - Push the drawer closed
        try:
            print(f"[Task] Pushing {target_drawer} closed.")
            execute_push(env, task, target_drawer, drawer_area)
        except Exception as e:
            print("[Error] Failed to push drawer closed:", e)

        # Step 5: Return to ready-pose at end
        try:
            print("[Task] Returning to ready-pose.")
            execute_go(env, task, drawer_area, ready_pose)
        except Exception as e:
            print("[Error] Failed to return to ready-pose:", e)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
