# run_skeleton_task.py (Completed with Exploration Phase and Predicate Validation)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates that the code does not validate if the object being placed into a drawer
        # is present in the object list. We need to explore and identify which predicate is missing.
        # We'll use the available exploration actions to check for object identification.

        # Suppose we have a list of all objects and drawers from the environment descriptions
        # For demonstration, we extract them from the positions dictionary
        all_objects = [obj for obj in positions.keys() if not obj.startswith('drawer')]
        all_drawers = [obj for obj in positions.keys() if obj.startswith('drawer')]

        # We'll keep track of identified objects
        identified_objects = set()

        # Try to identify all objects at their locations using the exploration skill
        for obj in all_objects:
            try:
                obj_pos = positions[obj]
                # Find the nearest location (for simulation, assume obj_pos is a location name or coordinates)
                # For this example, we assume obj_pos is a location name
                # Use the exploration skill: execute_go_identify
                # Since we don't have robot name and from-location, we use placeholders
                robot_name = 'robot'
                from_location = 'ready-pose'
                to_location = obj_pos if isinstance(obj_pos, str) else 'unknown-location'
                # Call the skill to identify the object
                obs, reward, done = execute_go_identify(
                    env,
                    task,
                    robot_name,
                    from_location,
                    to_location
                )
                identified_objects.add(obj)
                print(f"[Exploration] Identified object: {obj} at {to_location}")
            except Exception as e:
                print(f"[Exploration] Failed to identify object {obj}: {e}")

        # After exploration, we know which objects are present and identified
        # This helps us validate before placing any object into a drawer

        # === Main Task Plan ===
        # For demonstration, we attempt to place each identified object into a drawer if possible
        for drawer in all_drawers:
            try:
                # For each drawer, get the list of objects to place (simulate with all identified objects)
                for obj in identified_objects:
                    # Validate: Is the object present in the identified list?
                    if obj not in identified_objects:
                        print(f"[Validation] Object {obj} not identified, skipping placement.")
                        continue

                    # Additional validation: Is the drawer empty and unlocked?
                    # For this example, we assume we can check drawer state via positions or task state
                    # (In a real implementation, you would query the environment or task for these predicates)
                    drawer_state = positions.get(drawer, {})
                    is_unlocked = drawer_state.get('unlocked', True)
                    is_empty = drawer_state.get('empty', True)
                    if not is_unlocked:
                        print(f"[Validation] Drawer {drawer} is locked, cannot place {obj}.")
                        continue
                    if not is_empty:
                        print(f"[Validation] Drawer {drawer} is not empty, cannot place {obj}.")
                        continue

                    # Move to the object's location
                    obj_location = positions[obj] if isinstance(positions[obj], str) else 'unknown-location'
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location,
                        obj_location
                    )

                    # Pick the object
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj,
                        obj_location
                    )

                    # Move to the drawer's location
                    drawer_location = positions[drawer] if isinstance(positions[drawer], str) else 'unknown-location'
                    obs, reward, done = execute_go(
                        env,
                        task,
                        obj_location,
                        drawer_location
                    )

                    # Place the object into the drawer
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        drawer_location
                    )
                    print(f"[Task] Placed {obj} into {drawer} at {drawer_location}")

            except Exception as e:
                print(f"[Task] Failed to place object in drawer {drawer}: {e}")

        # === End of Main Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
