# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Robust Object List Construction ===
        # Build a dictionary of all expected objects and check their presence
        expected_objects = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2",
            "trash",
            "trash_bin"
        ]
        object_list = {}
        for obj_name in expected_objects:
            if obj_name in positions and positions[obj_name] is not None:
                object_list[obj_name] = positions[obj_name]
            else:
                object_list[obj_name] = None

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for missing predicates
        # For this, we attempt to identify all objects at the robot's current location
        # and check if any expected predicate (e.g., 'identified') is missing.

        # Find robot's current location (assume 'robot-at table1' from feedback)
        robot_location = None
        for key in positions:
            if key.startswith("robot") or key == "robot":
                robot_location = positions[key]
                break
        # If not found, fallback to a default location if available
        if robot_location is None and "table1" in positions:
            robot_location = positions["table1"]

        # For this example, let's assume the robot starts at 'table1'
        # and we want to identify all objects at this location
        # (In a real scenario, you would parse the observation/init state for this info)

        # Exploration: Try to identify all objects at the robot's location
        # (simulate the 'identified' predicate)
        identified_objects = []
        for obj_name, obj_pos in object_list.items():
            if obj_pos is not None and robot_location is not None:
                # Simple proximity check (within 0.2m)
                if np.linalg.norm(np.array(obj_pos) - np.array(robot_location)) < 0.2:
                    identified_objects.append(obj_name)
        if not identified_objects:
            print("[Exploration] No objects found at robot's location for identification.")
        else:
            print(f"[Exploration] Objects at robot's location ({robot_location}): {identified_objects}")
            # Here, in a real exploration, you would call a skill like execute_go_identify if available
            # Since only the provided skills are allowed, and 'execute_go' is available, we could move
            # For demonstration, we just print the result

        # === Main Task Plan Execution ===
        # Example: Try to pick and store dice1 if present

        # 1. Check if dice1 is present
        if object_list["dice1"] is not None:
            dice1_pos = object_list["dice1"]
            # 2. Move to dice1's position if not already there
            # (Assume we have a function or skill 'execute_go' that moves the robot)
            try:
                # Find current robot position key
                robot_pos_key = None
                for key, pos in positions.items():
                    if key.startswith("robot") or key == "robot":
                        robot_pos_key = key
                        break
                if robot_pos_key is not None:
                    current_robot_pos = positions[robot_pos_key]
                else:
                    current_robot_pos = robot_location
                # If not at dice1, move there
                if np.linalg.norm(np.array(current_robot_pos) - np.array(dice1_pos)) > 0.05:
                    print(f"[Task] Moving to dice1 at {dice1_pos}")
                    obs, reward, done, info = execute_go(env, task, from_pos=current_robot_pos, to_pos=dice1_pos)
                    if done:
                        print("[Task] Task ended after move!")
                        return
            except Exception as e:
                print(f"[Error] Exception during execute_go to dice1: {e}")

            # 3. Pick up dice1
            try:
                print("[Task] Attempting to pick up dice1")
                obs, reward, done, info = execute_pick(env, task, obj_name="dice1", obj_pos=dice1_pos)
                if done:
                    print("[Task] Task ended after picking up dice1!")
                    return
            except Exception as e:
                print(f"[Error] Exception during execute_pick for dice1: {e}")

            # 4. Move to drawer_top_place_left if present
            if object_list["drawer_top_place_left"] is not None:
                drawer_pos = object_list["drawer_top_place_left"]
                try:
                    print(f"[Task] Moving to drawer_top_place_left at {drawer_pos}")
                    obs, reward, done, info = execute_go(env, task, from_pos=dice1_pos, to_pos=drawer_pos)
                    if done:
                        print("[Task] Task ended after move to drawer!")
                        return
                except Exception as e:
                    print(f"[Error] Exception during execute_go to drawer: {e}")

                # 5. Place dice1 in drawer
                try:
                    print("[Task] Attempting to place dice1 in drawer_top_place_left")
                    obs, reward, done, info = execute_place(env, task, obj_name="dice1", place_pos=drawer_pos)
                    if done:
                        print("[Task] Task ended after placing dice1!")
                        return
                except Exception as e:
                    print(f"[Error] Exception during execute_place for dice1: {e}")
            else:
                print("[Task] drawer_top_place_left not found in object list. Cannot place dice1.")
        else:
            print("[Task] dice1 not found in object list. Skipping pick and place.")

        # === Additional Error Handling and Reporting ===
        # If any object is missing, report it
        missing_objects = [k for k, v in object_list.items() if v is None]
        if missing_objects:
            print(f"[Warning] The following expected objects were not found in the environment: {missing_objects}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
