# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate (Force Calibration) ===
        # The feedback indicates we need to calibrate the force required to pick up the dice.
        # We will use the available skills to explore the object's properties (e.g., weight, durability)
        # and simulate a force calibration step.

        # For demonstration, we assume the dice is named 'dice' and the drawer is 'drawer'
        # and both are present in the positions dictionary.
        # You may need to adjust these names based on your environment's object naming.

        dice_name = None
        drawer_name = None
        for obj in positions:
            if 'dice' in obj.lower():
                dice_name = obj
            if 'drawer' in obj.lower():
                drawer_name = obj

        if dice_name is None or drawer_name is None:
            print("[Error] Could not find 'dice' or 'drawer' in object positions.")
            shutdown_environment(env)
            return

        dice_pos = positions[dice_name]
        drawer_pos = positions[drawer_name]

        # Assume the robot starts at a known location; get its current location
        robot_location = None
        for obj in positions:
            if 'robot' in obj.lower():
                robot_location = positions[obj]
                break

        # If robot location is not available, set to a default or skip
        if robot_location is None:
            robot_location = (0, 0, 0)  # Default/fallback

        # === Step 1: Move to Dice Location ===
        try:
            print(f"[Exploration] Moving to dice at {dice_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=dice_pos,
                max_steps=100
            )
            robot_location = dice_pos
        except Exception as e:
            print(f"[Error] Failed to move to dice: {e}")
            shutdown_environment(env)
            return

        # === Step 2: Calibrate Force (Weight/Grip) ===
        # Use exploration skills to determine weight and durability
        # (simulate force calibration by picking with weight/durability sensing)
        force_calibrated = False
        try:
            print("[Exploration] Calibrating force by picking dice and sensing weight/durability")
            # Try to pick and sense weight
            obs, reward, done = execute_pick_weight(
                env,
                task,
                robot='robot',  # Replace with actual robot name if needed
                obj=dice_name,
                loc=dice_pos,
                max_steps=100
            )
            # Try to pick and sense durability (if needed)
            obs, reward, done = execute_pick_durability(
                env,
                task,
                robot='robot',
                obj=dice_name,
                loc=dice_pos,
                max_steps=100
            )
            force_calibrated = True
            print("[Exploration] Force calibration complete: weight and durability known.")
        except Exception as e:
            print(f"[Error] Force calibration failed: {e}")
            shutdown_environment(env)
            return

        # === Step 3: Move to Drawer Location ===
        try:
            print(f"[Task] Moving to drawer at {drawer_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_pos,
                max_steps=100
            )
            robot_location = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            shutdown_environment(env)
            return

        # === Step 4: Open Drawer (if required) ===
        # If the drawer is locked, unlock and open it using available skills
        # For simplicity, we assume the handle is named 'handle' and is at the drawer location
        handle_name = None
        for obj in positions:
            if 'handle' in obj.lower():
                handle_name = obj
                break

        if handle_name is not None:
            try:
                print(f"[Task] Picking handle '{handle_name}' at {drawer_pos}")
                obs, reward, done = execute_pick_handle(
                    env,
                    task,
                    h=handle_name,
                    p=drawer_pos,
                    max_steps=100
                )
                print(f"[Task] Pulling drawer '{drawer_name}' with handle '{handle_name}'")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    d=drawer_name,
                    h=handle_name,
                    p=drawer_pos,
                    max_steps=100
                )
            except Exception as e:
                print(f"[Error] Failed to open drawer: {e}")
                shutdown_environment(env)
                return
        else:
            print("[Warning] No handle found for drawer; skipping pull action.")

        # === Step 5: Place Dice in Drawer ===
        try:
            print(f"[Task] Placing dice '{dice_name}' in drawer '{drawer_name}' at {drawer_pos}")
            obs, reward, done = execute_place_object(
                env,
                task,
                o=dice_name,
                d=drawer_name,
                p=drawer_pos,
                max_steps=100
            )
            print("[Task] Dice placed in drawer successfully.")
        except Exception as e:
            print(f"[Error] Failed to place dice in drawer: {e}")
            shutdown_environment(env)
            return

        # === Step 6: Push Drawer Closed (if required) ===
        try:
            print(f"[Task] Pushing drawer '{drawer_name}' closed at {drawer_pos}")
            obs, reward, done = execute_push(
                env,
                task,
                d=drawer_name,
                p=drawer_pos,
                max_steps=100
            )
            print("[Task] Drawer closed successfully.")
        except Exception as e:
            print(f"[Warning] Could not push drawer closed: {e}")

        # === Step 7: Return to Ready Pose ===
        try:
            print(f"[Task] Returning to ready pose from {robot_location}")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location,
                max_steps=100
            )
            print("[Task] Robot returned to ready pose.")
        except Exception as e:
            print(f"[Warning] Could not return to ready pose: {e}")

        print("===== Task Completed Successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()