# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Retrieve object names from the environment ===
        # This assumes the environment or task provides a list of object names
        # If not, you may need to extract them from descriptions or positions
        try:
            object_names = list(positions.keys())
        except Exception as e:
            print("[Error] Could not retrieve object names from positions:", e)
            object_names = []

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to ensure that the object (e.g., "dice1") is present before interacting
        # We'll perform an exploration step to identify all objects at each location

        # For demonstration, let's assume we want to store "dice1" in a drawer
        target_object = "dice1"
        drawer_name = None
        handle_name = None
        robot_location = None

        # Find the drawer and handle names from object_names (simple heuristics)
        for name in object_names:
            if "drawer" in name and drawer_name is None:
                drawer_name = name
            if "handle" in name and handle_name is None:
                handle_name = name

        # Find robot's current location (if available)
        # This may require parsing the observation or using a default
        # For now, we use a default or the first location in positions
        location_names = [k for k in object_names if "location" in k or "pose" in k]
        if location_names:
            robot_location = location_names[0]
        else:
            robot_location = "ready-pose"

        # === Validation Check: Ensure Target Object Exists ===
        if target_object not in object_names:
            print(f"[Error] Target object '{target_object}' not found in object list: {object_names}")
            print("[Exploration] Aborting task as required object is missing.")
            return
        else:
            print(f"[Exploration] Target object '{target_object}' found. Proceeding with task.")

        # === (Optional) Force Calibration and Safety Precautions ===
        # Not implemented here, but could be added as part of skill_code or environment setup

        # === Task Plan Execution ===
        # 1. Move to the object's location (if not already there)
        # 2. Pick the object (execute_pick)
        # 3. Move to the drawer location
        # 4. Open the drawer (execute_pick_handle + execute_pull)
        # 5. Place the object in the drawer (execute_place)
        # 6. Close the drawer (execute_push)
        # 7. Return to ready pose (execute_go_ready)

        # For all actions, handle exceptions gracefully

        try:
            # 1. Move to the object's location
            obj_pos = positions.get(target_object, None)
            if obj_pos is None:
                print(f"[Error] Could not find position for {target_object}.")
                return

            # Assume robot starts at 'ready-pose'
            print(f"[Task] Moving to {target_object} at position {obj_pos}")
            obs, reward, done = execute_go(env, task, from_location="ready-pose", to_location=target_object)
            if done:
                print("[Task] Task ended after moving to object!")
                return

            # 2. Pick the object
            print(f"[Task] Picking up {target_object}")
            obs, reward, done = execute_pick(env, task, object_name=target_object, location=target_object)
            if done:
                print("[Task] Task ended after picking object!")
                return

            # 3. Move to the drawer location
            if drawer_name is None:
                print("[Error] Drawer not found in object list.")
                return
            drawer_pos = positions.get(drawer_name, None)
            if drawer_pos is None:
                print(f"[Error] Could not find position for {drawer_name}.")
                return
            print(f"[Task] Moving to drawer {drawer_name} at position {drawer_pos}")
            obs, reward, done = execute_go(env, task, from_location=target_object, to_location=drawer_name)
            if done:
                print("[Task] Task ended after moving to drawer!")
                return

            # 4. Open the drawer (pick handle + pull)
            if handle_name is None:
                print("[Error] Handle not found in object list.")
                return
            print(f"[Task] Picking handle {handle_name}")
            obs, reward, done = execute_pick(env, task, object_name=handle_name, location=drawer_name)
            if done:
                print("[Task] Task ended after picking handle!")
                return

            print(f"[Task] Pulling drawer {drawer_name} using handle {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=drawer_name)
            if done:
                print("[Task] Task ended after pulling drawer!")
                return

            # 5. Place the object in the drawer
            print(f"[Task] Placing {target_object} in {drawer_name}")
            obs, reward, done = execute_place(env, task, object_name=target_object, drawer_name=drawer_name, location=drawer_name)
            if done:
                print("[Task] Task ended after placing object!")
                return

            # 6. Close the drawer
            print(f"[Task] Pushing drawer {drawer_name} closed")
            obs, reward, done = execute_push(env, task, drawer_name=drawer_name, location=drawer_name)
            if done:
                print("[Task] Task ended after pushing drawer!")
                return

            # 7. Return to ready pose
            print("[Task] Returning to ready pose")
            obs, reward, done = execute_go(env, task, from_location=drawer_name, to_location="ready-pose")
            if done:
                print("[Task] Task ended after returning to ready pose!")
                return

            print("[Task] Task completed successfully.")

        except Exception as e:
            print(f"[Exception] Error during task execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
