# run_skeleton_task.py (Filled as per requirements)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # --- Example keys; these must match your env ---
        # We will try to infer required keys. You may update them if needed to match your env.
        # Let's assume: 'robot', 'drawer1', 'drawer_handle', 'target_object', 'ready-pose'
        # Provide graceful fallback if keys don't exist

        robot_loc = positions.get('robot', None)
        drawer_loc = positions.get('drawer1', None)
        handle_loc = positions.get('drawer_handle', None)
        object_loc = positions.get('target_object', None)
        ready_pose = positions.get('ready-pose', None)
        
        # Check for missing key info
        if None in [drawer_loc, handle_loc]:
            print("[Warning] Required object positions not found in get_object_positions(). Available keys:", list(positions.keys()))
        
        # === Exploration Phase: Detect Missing Predicate via execute_pull ===
        # The provided feedback suggests that (drawer-unlocked drawer1) is a missing predicate.
        # Thus, we conduct an exploration pull on the drawer handle to check for lock status,
        # using the available primitive skills and respecting their names.

        # Assumptions for names - update these according to your actual keys in object_positions()
        drawer_name = 'drawer1'
        handle_name = 'drawer_handle'
        pull_location = drawer_loc if drawer_loc else (0,0,0)
        # For many environments, the robot must first move near the handle/drawer and pick the handle before pulling.

        try:
            print("[Exploration] Moving robot to drawer location for exploration...")
            # Use execute_go primitive if available to move to the handle/drawer
            obs, reward, done = execute_go(env, task, from_location=robot_loc, to_location=drawer_loc)
            if done:
                print("[Exploration] Task finished early during move.")
                return

            print("[Exploration] Trying to pick the drawer handle to check lock.")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_loc)
            if done:
                print("[Exploration] Task finished early during pick-handle.")
                return

            print("[Exploration] Attempting to pull the drawer (explore lock status)...")
            # This skill both tests unlocking in some PDDL domains
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_loc)
            if done:
                print("[Exploration] Drawer pull completed; task finished.")
                return

            print("[Exploration] Exploration phase complete. Missing predicate was likely (drawer-unlocked drawer1).")

        except Exception as e:
            print(f"[Exploration] Exception occurred during exploration phase: {e}")

        # === Task-Oriented Execution (Oracle plan, example steps) ===

        try:
            # Return to ready-pose if post-place or after skill transition
            if ready_pose:
                print("[Task] Returning to ready-pose...")
                obs, reward, done = execute_go(env, task, from_location=drawer_loc, to_location=ready_pose)
                if done:
                    print("[Task] Task ended after returning to ready pose.")
                    return

            # Example: Move robot to the location of the target object
            if object_loc:
                print("[Task] Moving robot to target object location...")
                obs, reward, done = execute_go(env, task, from_location=ready_pose if ready_pose else robot_loc, to_location=object_loc)
                if done:
                    print("[Task] Finished after moving to object.")
                    return

                print("[Task] Picking up target object...")
                obs, reward, done = execute_pick(env, task, "target_object", object_loc)
                if done:
                    print("[Task] Finished after picking up the object.")
                    return

                # Move to drawer again for placing
                print("[Task] Moving to drawer to place the object...")
                obs, reward, done = execute_go(env, task, from_location=object_loc, to_location=drawer_loc)
                if done:
                    print("[Task] Finished after approaching drawer for placing.")
                    return

                print("[Task] Placing the object in the drawer...")
                obs, reward, done = execute_place(env, task, "target_object", drawer_name, drawer_loc)
                if done:
                    print("[Task] Object placed successfully. Task complete!")
                    return

            print("[Task] All steps in the example task completed.")
        except Exception as e:
            print(f"[Task] Exception encountered during task execution: {e}")

        # --- End of oracle plan / task steps ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()