# run_skeleton_task.py (Filled Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all skills as provided

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Setup recording for step and obs
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()

        # === Begin Exploration for Missing Predicate ===
        # Based on feedback: (robot-at drawer_bottom_place_left)
        print(">> [Exploration] Identifying predicates at drawer_bottom_place_left <<")

        # Find a location named 'drawer_bottom_place_left' in positions or in descriptions
        location_name = 'drawer_bottom_place_left'
        robot_pos = None
        if 'robot' in positions:
            robot_pos = positions['robot']
        elif 'robot_base' in positions:
            robot_pos = positions['robot_base']
        
        if location_name in positions:
            target_pos = positions[location_name]
        else:
            # Try parsing from descriptions or assume position if not present
            target_pos = None

        # --- Step 1: Move robot to drawer_bottom_place_left ---
        # Use 'execute_go' as per available skills
        # Try to figure out current location if possible, otherwise, use a placeholder
        current_location = None
        for key in positions:
            if key.startswith('robot_at_') and positions[key]:
                current_location = key.replace('robot_at_', '')
                break
        if not current_location:
            # Fallback to a default or extract from descriptions, or guess
            current_location = 'ready-pose'

        try:
            # Call execute_go to move from current to target location
            print(f"[Exploration] Moving robot from '{current_location}' to '{location_name}' using execute_go skill.")
            obs, reward, done = execute_go(env, task, from_location=current_location, to_location=location_name)
        except Exception as e:
            print(f"[Error] Failed to move robot: {e}")

        # --- Step 2: Attempt to identify predicate at location (simulate exploration) ---
        # Since exploration domain includes execute_go_identify etc., but we're limited to available skills,
        # we mimic this by moving the robot and then observing.
        print(f"[Exploration] Observation after moving to '{location_name}':")
        obs = task.get_observation()
        print(obs)

        # You would here check the predicates/state after movement to see if new information can be inferred.
        # For this code, just print out the current observable state.

        print(">> [Exploration] If predicate missing, inspect obs for clues.")

        # Optionally, continue with additional exploration actions as available (using the available skills):
        for skill_name in ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']:
            print(f"[Exploration] Attempting skill: {skill_name}")
            try:
                skill_func = globals().get(skill_name, None)
                if skill_func is None:
                    print(f" - [Skip] Skill {skill_name} not implemented in skill_code.")
                    continue

                # Prepare plausible generic parameters; specifics would depend on environment state
                # Here, make a best-effort to use positions/descriptions or fallback names

                # Example heuristic for skill parameterization
                if skill_name == 'execute_pick':
                    # Try to pick a visible object (not handle), at target location
                    objects = [k for k in positions.keys() if 'obj' in k or 'block' in k or 'item' in k]
                    if objects:
                        obj = objects[0]
                        print(f" - Trying to pick object: {obj} at {location_name}")
                        obs, reward, done = skill_func(env, task, obj, location_name)
                elif skill_name == 'execute_place':
                    drawers = [k for k in positions if 'drawer' in k]
                    objects = [k for k in positions if 'obj' in k or 'block' in k or 'item' in k]
                    if objects and drawers:
                        obj = objects[0]
                        drawer = drawers[0]
                        print(f" - Trying to place object: {obj} in {drawer} at {location_name}")
                        obs, reward, done = skill_func(env, task, obj, drawer, location_name)
                elif skill_name == 'execute_push':
                    drawers = [k for k in positions if 'drawer' in k]
                    if drawers:
                        drawer = drawers[0]
                        print(f" - Trying to push drawer: {drawer} at {location_name}")
                        obs, reward, done = skill_func(env, task, drawer, location_name)
                elif skill_name == 'execute_pull':
                    drawers = [k for k in positions if 'drawer' in k]
                    handles = [k for k in positions if 'handle' in k]
                    if drawers and handles:
                        drawer = drawers[0]
                        handle = handles[0]
                        print(f" - Trying to pull drawer: {drawer} with handle: {handle} at {location_name}")
                        obs, reward, done = skill_func(env, task, drawer, handle, location_name)
                elif skill_name == 'execute_sweep':
                    objects = [k for k in positions if 'obj' in k or 'block' in k or 'item' in k]
                    if objects:
                        obj = objects[0]
                        print(f" - Trying to sweep object: {obj} at {location_name}")
                        obs, reward, done = skill_func(env, task, obj, location_name)
                elif skill_name == 'execute_rotate':
                    # If this requires objects, attempt
                    print(f" - Trying to rotate at {location_name}")
                    obs, reward, done = skill_func(env, task, location_name)
                elif skill_name == 'execute_go':
                    print(f" - Trying to re-move from {location_name} to ready-pose")
                    obs, reward, done = skill_func(env, task, from_location=location_name, to_location='ready-pose')
                elif skill_name == 'execute_gripper':
                    print(f" - Trying to execute gripper")
                    obs, reward, done = skill_func(env, task)
                else:
                    print(f" - Unknown parameterization for skill {skill_name}")

                # Print obs after skill execution for debugging
                print(f" - Observation after skill {skill_name}:")
                print(obs)

            except Exception as e:
                print(f" - [Error] Skill {skill_name} failed: {e}")

        print(">> [Exploration] Exploration phase complete. Review observations for missing predicates.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()