# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore which predicate is missing (e.g., on-object, on-location)
        # We'll use available skills to explore the environment and log the results.

        # For demonstration, let's try to identify all objects at each location
        # and check if 'on-object' and 'on-location' predicates are needed.

        # Assume we have a list of objects and locations from the observation or positions
        # For this example, we try to infer them from positions dict
        object_names = []
        location_names = set()
        for name, pos in positions.items():
            if "drawer" in name or "floor" in name or "area" in name or "pose" in name:
                location_names.add(name)
            else:
                object_names.append(name)
        location_names = list(location_names)

        # If not found, fallback to some defaults (for robustness)
        if not object_names:
            object_names = ['dice1', 'dice2', 'handle_bottom', 'handle_middle', 'handle_top']
        if not location_names:
            location_names = ['floor', 'drawer_bottom', 'drawer_middle', 'drawer_top', 'ready-pose']

        # Exploration: Try to identify objects at each location using available skills
        # Use execute_go (move), execute_pick, execute_pull, etc.

        # Start at ready-pose
        current_location = 'ready-pose'
        robot_free = True
        hand_empty = True

        # Try to move to each location and "identify" objects (simulate exploration)
        for loc in location_names:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Moving from {current_location} to {loc}")
                obs, reward, done = execute_go(env, task, from_location=current_location, to_location=loc)
                current_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

            # Try to pick up any object at this location (simulate 'on-object'/'on-location' check)
            for obj in object_names:
                try:
                    # Try picking as object (not handle)
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, object_name=obj, location_name=loc)
                    print(f"[Exploration] Picked {obj} at {loc}")
                    # Place it back if possible (simulate place)
                    for d in location_names:
                        if "drawer" in d:
                            try:
                                print(f"[Exploration] Attempting to place {obj} in {d} at {loc}")
                                obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=d, location_name=loc)
                                print(f"[Exploration] Placed {obj} in {d} at {loc}")
                                break
                            except Exception as e:
                                continue
                    # After place, break to next object
                    break
                except Exception as e:
                    # Try picking as handle if object is a handle
                    try:
                        print(f"[Exploration] Attempting to pick handle {obj} at {loc}")
                        obs, reward, done = execute_pick_handle(env, task, handle_name=obj, location_name=loc)
                        print(f"[Exploration] Picked handle {obj} at {loc}")
                        # Try to pull a drawer with this handle
                        for d in location_names:
                            if "drawer" in d:
                                try:
                                    print(f"[Exploration] Attempting to pull {d} with handle {obj} at {loc}")
                                    obs, reward, done = execute_pull(env, task, drawer_name=d, handle_name=obj, location_name=loc)
                                    print(f"[Exploration] Pulled {d} with handle {obj} at {loc}")
                                    break
                                except Exception as e:
                                    continue
                        break
                    except Exception as e2:
                        continue

        # After exploration, print a message about missing predicates
        print("[Exploration] Exploration complete. If errors occurred when picking or placing, check if 'on-object' or 'on-location' predicates are missing in your PDDL or observation.")

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, let's assume a simple plan:
        # 1. Move to floor, pick up dice1, move to drawer_bottom, place dice1 in drawer_bottom, return to ready-pose

        try:
            # Move to floor
            if current_location != 'floor':
                print("[Task] Moving to floor")
                obs, reward, done = execute_go(env, task, from_location=current_location, to_location='floor')
                current_location = 'floor'

            # Pick up dice1
            print("[Task] Picking up dice1 at floor")
            obs, reward, done = execute_pick(env, task, object_name='dice1', location_name='floor')

            # Move to drawer_bottom
            print("[Task] Moving to drawer_bottom")
            obs, reward, done = execute_go(env, task, from_location='floor', to_location='drawer_bottom')
            current_location = 'drawer_bottom'

            # Place dice1 in drawer_bottom
            print("[Task] Placing dice1 in drawer_bottom")
            obs, reward, done = execute_place(env, task, object_name='dice1', drawer_name='drawer_bottom', location_name='drawer_bottom')

            # Return to ready-pose
            print("[Task] Returning to ready-pose")
            obs, reward, done = execute_go_ready(env, task, from_location='drawer_bottom')
            current_location = 'ready-pose'

            print("[Task] Oracle plan executed successfully.")

        except Exception as e:
            print(f"[Task] Error during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
