# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Retrieve object and drawer lists from the environment ===
        # This assumes the environment or task provides these lists.
        # If not, you may need to parse them from descriptions or positions.
        try:
            object_list = task.get_object_names() if hasattr(task, "get_object_names") else list(positions.keys())
        except Exception:
            object_list = list(positions.keys())
        try:
            drawer_list = task.get_drawer_names() if hasattr(task, "get_drawer_names") else [k for k in positions.keys() if "drawer" in k]
        except Exception:
            drawer_list = [k for k in positions.keys() if "drawer" in k]

        # === EXPLORATION PHASE: Identify missing predicate (e.g., lock-known) ===
        # The exploration domain suggests we may need to check for lock status before pulling a drawer.
        # We'll attempt to "pull" the drawer and, if it fails, try to "explore" (e.g., by using execute_pull for lock-known).
        # For demonstration, we will try to identify all objects at each location.

        # For each location, identify objects (simulate exploration)
        for loc in set([positions[obj][0] for obj in positions]):
            # Try to identify all objects at this location
            for obj in object_list:
                if obj in positions and positions[obj][0] == loc:
                    try:
                        # Use execute_go (move) to the location if not already there
                        robot_pos = task.get_robot_location() if hasattr(task, "get_robot_location") else None
                        if robot_pos != loc:
                            obs, reward, done, info = execute_go(env, task, from_location=robot_pos, to_location=loc)
                        # Simulate identification (if such a skill exists)
                        # Here, we just print for demonstration
                        print(f"[Exploration] Identified object '{obj}' at location '{loc}'")
                    except Exception as e:
                        print(f"[Exploration] Error identifying object '{obj}' at '{loc}': {e}")

        # === MAIN TASK PHASE ===
        # Example: Pick up an object and place it in a drawer, with validation

        # Select an object to pick and a drawer to place it in
        # (Replace with actual logic or plan as needed)
        selected_object = None
        for obj in object_list:
            if "handle" not in obj and "drawer" not in obj:
                selected_object = obj
                break
        selected_drawer = None
        for dr in drawer_list:
            if "drawer" in dr:
                selected_drawer = dr
                break

        # Validate object and drawer selection
        if selected_object is None:
            print("[Task] No suitable object found to pick.")
            return
        if selected_drawer is None:
            print("[Task] No suitable drawer found to place the object.")
            return

        # Check if the object is in the object list before attempting to put it into the drawer
        if selected_object not in object_list:
            print(f"[Validation] Object '{selected_object}' not in object list. Aborting action.")
            return

        # Move to the object's location
        obj_location = positions[selected_object][0] if selected_object in positions else None
        if obj_location is None:
            print(f"[Task] Could not determine location for object '{selected_object}'.")
            return

        try:
            robot_pos = task.get_robot_location() if hasattr(task, "get_robot_location") else None
            if robot_pos != obj_location:
                obs, reward, done, info = execute_go(env, task, from_location=robot_pos, to_location=obj_location)
        except Exception as e:
            print(f"[Task] Error moving to object location: {e}")
            return

        # Pick the object
        try:
            obs, reward, done, info = execute_pick(env, task, object_name=selected_object, location=obj_location)
            print(f"[Task] Picked up object '{selected_object}' at '{obj_location}'.")
        except Exception as e:
            print(f"[Task] Error picking object '{selected_object}': {e}")
            return

        # Move to the drawer's location
        drawer_location = positions[selected_drawer][0] if selected_drawer in positions else None
        if drawer_location is None:
            print(f"[Task] Could not determine location for drawer '{selected_drawer}'.")
            return

        try:
            robot_pos = task.get_robot_location() if hasattr(task, "get_robot_location") else None
            if robot_pos != drawer_location:
                obs, reward, done, info = execute_go(env, task, from_location=robot_pos, to_location=drawer_location)
        except Exception as e:
            print(f"[Task] Error moving to drawer location: {e}")
            return

        # Open the drawer if needed (simulate pull with handle if required)
        # For demonstration, try to pull the drawer using a handle if available
        handle_name = None
        for obj in object_list:
            if "handle" in obj and obj in positions and positions[obj][0] == drawer_location:
                handle_name = obj
                break
        if handle_name:
            try:
                obs, reward, done, info = execute_pick(env, task, object_name=handle_name, location=drawer_location)
                print(f"[Task] Picked up handle '{handle_name}' at '{drawer_location}'.")
                obs, reward, done, info = execute_pull(env, task, drawer_name=selected_drawer, handle_name=handle_name, location=drawer_location)
                print(f"[Task] Pulled open drawer '{selected_drawer}' with handle '{handle_name}'.")
            except Exception as e:
                print(f"[Task] Error pulling drawer '{selected_drawer}' with handle '{handle_name}': {e}")

        # Place the object in the drawer
        try:
            obs, reward, done, info = execute_place(env, task, object_name=selected_object, drawer_name=selected_drawer, location=drawer_location)
            print(f"[Task] Placed object '{selected_object}' in drawer '{selected_drawer}'.")
        except Exception as e:
            print(f"[Task] Error placing object '{selected_object}' in drawer '{selected_drawer}': {e}")
            return

        # Optionally, push the drawer closed
        try:
            obs, reward, done, info = execute_push(env, task, drawer_name=selected_drawer, location=drawer_location)
            print(f"[Task] Pushed drawer '{selected_drawer}' closed.")
        except Exception as e:
            print(f"[Task] Error pushing drawer '{selected_drawer}': {e}")

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()