# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names as per the PDDL and feedback ===
        drawers = ['drawer_top', 'drawer_middle', 'drawer_bottom']
        handles = ['handle_top', 'handle_middle', 'handle_bottom']
        dice = ['dice1', 'dice2']
        locations = ['floor', 'drawer-area', 'ready-pose']

        # Map handles to drawers
        handle_to_drawer = {
            'handle_top': 'drawer_top',
            'handle_middle': 'drawer_middle',
            'handle_bottom': 'drawer_bottom'
        }

        # === Exploration Phase: Identify missing predicate (lock-known) ===
        # The feedback and exploration domain suggest that the robot must determine the lock status of drawers.
        # We'll attempt to pull each handle and see if the drawer is locked or unlocked.
        # This will set the 'lock-known' predicate for each drawer.

        print("[Exploration] Checking lock status of each drawer by pulling handles...")

        for handle in handles:
            drawer = handle_to_drawer[handle]
            try:
                # Move to handle location (assume all handles are on the floor at the start)
                handle_pos = positions.get(handle, positions.get('floor', None))
                if handle_pos is None:
                    print(f"[Exploration] Could not find position for {handle}, skipping.")
                    continue

                # Move robot to handle location if not already there
                robot_pos = positions.get('robot', positions.get('ready-pose', None))
                if robot_pos != handle_pos:
                    try:
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location='ready-pose',
                            to_location='floor'
                        )
                    except Exception as e:
                        print(f"[Exploration] Error moving to handle {handle}: {e}")
                        continue

                # Pick up the handle
                try:
                    obs, reward, done = execute_pick_handle(
                        env,
                        task,
                        h=handle,
                        p='floor'
                    )
                except Exception as e:
                    print(f"[Exploration] Error picking handle {handle}: {e}")
                    continue

                # Try to pull the drawer using the handle
                try:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        d=drawer,
                        h=handle,
                        p='floor'
                    )
                    print(f"[Exploration] Pulled {drawer} with {handle}: Drawer is UNLOCKED.")
                except Exception as e:
                    print(f"[Exploration] Could not pull {drawer} with {handle}: Drawer is LOCKED.")

                # Place handle back on the floor (if needed)
                # (No explicit place_handle in skills, so skip)

                # Return to ready-pose after each attempt
                try:
                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        from_location='floor'
                    )
                except Exception as e:
                    print(f"[Exploration] Error returning to ready-pose: {e}")

            except Exception as e:
                print(f"[Exploration] Unexpected error with handle {handle}: {e}")

        print("[Exploration] Drawer lock status exploration complete.")

        # === Main Task Plan: Place dice into drawers ===
        # For each dice, pick it up and place it into an available, unlocked, open drawer.

        # We'll use only unlocked drawers (top and bottom, per feedback).
        available_drawers = ['drawer_top', 'drawer_bottom']
        dice_to_place = dice.copy()

        for i, die in enumerate(dice_to_place):
            # 1. Move to dice location (assume on floor)
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='ready-pose',
                    to_location='floor'
                )
            except Exception as e:
                print(f"[Task] Error moving to dice {die}: {e}")
                continue

            # 2. Pick up the dice
            try:
                obs, reward, done = execute_pick_object(
                    env,
                    task,
                    o=die,
                    p='floor'
                )
            except Exception as e:
                print(f"[Task] Error picking up {die}: {e}")
                continue

            # 3. Return to ready-pose (required by need-ready flag)
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location='floor'
                )
            except Exception as e:
                print(f"[Task] Error returning to ready-pose after picking {die}: {e}")

            # 4. Move to drawer area
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='ready-pose',
                    to_location='drawer-area'
                )
            except Exception as e:
                print(f"[Task] Error moving to drawer-area for {die}: {e}")
                continue

            # 5. Open the drawer (if not already open)
            drawer = available_drawers[i % len(available_drawers)]
            handle = [h for h, d in handle_to_drawer.items() if d == drawer][0]
            try:
                # Pick up the handle (assume on floor)
                obs, reward, done = execute_pick_handle(
                    env,
                    task,
                    h=handle,
                    p='floor'
                )
            except Exception as e:
                print(f"[Task] Error picking handle {handle} for {drawer}: {e}")
                continue

            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    d=drawer,
                    h=handle,
                    p='drawer-area'
                )
            except Exception as e:
                print(f"[Task] Error pulling {drawer} with {handle}: {e}")
                continue

            # 6. Place the dice into the drawer
            try:
                obs, reward, done = execute_place_object(
                    env,
                    task,
                    o=die,
                    d=drawer,
                    p='drawer-area'
                )
            except Exception as e:
                print(f"[Task] Error placing {die} into {drawer}: {e}")
                continue

            # 7. Return to ready-pose (required by need-ready flag)
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location='drawer-area'
                )
            except Exception as e:
                print(f"[Task] Error returning to ready-pose after placing {die}: {e}")

            # 8. Push the drawer closed
            try:
                obs, reward, done = execute_push(
                    env,
                    task,
                    d=drawer,
                    p='drawer-area'
                )
            except Exception as e:
                print(f"[Task] Error pushing {drawer} closed: {e}")

            # 9. Return to ready-pose
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location='drawer-area'
                )
            except Exception as e:
                print(f"[Task] Error returning to ready-pose after closing {drawer}: {e}")

        print("[Task] All dice placed in drawers.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
