# run_skeleton_task.py (Completed with Exploration and Safety Checks)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        object_names = list(positions.keys())

        # === Exploration Phase: Identify Missing Predicate or Object ===
        # Feedback indicates possible mismatch between declared and actual objects (e.g., "dice1" missing)
        # Also, need to check for object presence, drawer state, and force calibration

        # Example: Suppose the plan expects to manipulate "dice1"
        expected_object = "dice1"
        drawer_name = "drawer1"  # Example; adjust as per your environment
        robot_location = None
        drawer_location = None

        # 1. Safety Check: Is the expected object present?
        if expected_object not in object_names:
            print(f"[Error] Expected object '{expected_object}' not found in environment objects: {object_names}")
            print("[Exploration] Attempting to identify available objects and their properties...")

            # Exploration: Identify all objects at each location
            for loc in set([v for v in positions.values()]):
                for obj in object_names:
                    try:
                        # Use execute_go_identify to identify objects at location
                        obs, reward, done = execute_go_identify(
                            env, task, from_location=None, to_location=loc
                        )
                        print(f"[Exploration] Identified objects at {loc}: {obj}")
                    except Exception as e:
                        print(f"[Exploration] Could not identify object {obj} at {loc}: {e}")
            print("[Exploration] Please update the plan or object list to match the environment.")
            return

        # 2. Find object and drawer locations
        object_pos = positions.get(expected_object, None)
        if object_pos is None:
            print(f"[Error] Could not find position for object '{expected_object}'.")
            return

        # Try to find the drawer location
        for name in object_names:
            if "drawer" in name:
                drawer_name = name
                drawer_location = positions[drawer_name]
                break
        if drawer_location is None:
            print("[Error] No drawer found in environment.")
            return

        # 3. Move robot to object location if not already there
        # (Assume robot's current location is available in obs or via a helper)
        try:
            robot_location = obs.get('robot_location', None)
        except Exception:
            robot_location = None
        if robot_location != object_pos:
            try:
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=object_pos)
                print(f"[Task] Robot moved to object location: {object_pos}")
            except Exception as e:
                print(f"[Error] Failed to move robot to object location: {e}")
                return

        # 4. Force Calibration: (Simulated via weight-known predicate)
        # Use execute_pick_weight to calibrate force and pick up the object
        try:
            obs, reward, done = execute_pick_weight(env, task, obj=expected_object, loc=object_pos)
            print(f"[Task] Picked up '{expected_object}' with force calibration (weight-known).")
        except Exception as e:
            print(f"[Error] Failed to pick up '{expected_object}' with force calibration: {e}")
            return

        # 5. Move to drawer location
        try:
            obs, reward, done = execute_go(env, task, from_location=object_pos, to_location=drawer_location)
            print(f"[Task] Robot moved to drawer location: {drawer_location}")
        except Exception as e:
            print(f"[Error] Failed to move robot to drawer location: {e}")
            return

        # 6. Safety Check: Is the drawer open and not full?
        # (Assume we can check via obs or a helper; if not, try to open)
        drawer_open = obs.get(f"{drawer_name}_open", False)
        drawer_full = obs.get(f"{drawer_name}_full", False)
        if not drawer_open:
            # Try to open the drawer (requires handle, etc.)
            print(f"[Task] Drawer '{drawer_name}' is not open. Attempting to open...")
            # Find handle for the drawer
            handle_name = None
            for name in object_names:
                if "handle" in name and drawer_name in name:
                    handle_name = name
                    break
            if handle_name is None:
                print(f"[Error] No handle found for drawer '{drawer_name}'.")
                return
            handle_pos = positions[handle_name]
            # Move to handle location if needed
            if drawer_location != handle_pos:
                try:
                    obs, reward, done = execute_go(env, task, from_location=drawer_location, to_location=handle_pos)
                    print(f"[Task] Robot moved to handle location: {handle_pos}")
                except Exception as e:
                    print(f"[Error] Failed to move to handle location: {e}")
                    return
            # Pick handle
            try:
                obs, reward, done = execute_pick(env, task, obj=handle_name, loc=handle_pos)
                print(f"[Task] Picked up handle '{handle_name}'.")
            except Exception as e:
                print(f"[Error] Failed to pick up handle '{handle_name}': {e}")
                return
            # Pull to open drawer
            try:
                obs, reward, done = execute_pull(env, task, d=drawer_name, h=handle_name, p=handle_pos)
                print(f"[Task] Pulled handle to open drawer '{drawer_name}'.")
            except Exception as e:
                print(f"[Error] Failed to pull handle to open drawer: {e}")
                return

        if drawer_full:
            print(f"[Error] Drawer '{drawer_name}' is full. Cannot place object.")
            return

        # 7. Place object in drawer
        try:
            obs, reward, done = execute_place(env, task, o=expected_object, d=drawer_name, p=drawer_location)
            print(f"[Task] Placed '{expected_object}' in drawer '{drawer_name}'.")
        except Exception as e:
            print(f"[Error] Failed to place '{expected_object}' in drawer: {e}")
            return

        # 8. Error Handling and Feedback
        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
