# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Retrieve Object List from Observation ===
        # The observation (init) is assumed to be available as a string or structure
        # For this example, we will define the object list as per the feedback
        object_list = [
            'drawer_top_handle', 'drawer_middle_handle', 'drawer_bottom_handle',
            'drawer_top_place_left', 'drawer_top_place_right', 'drawer_middle_place',
            'drawer_bottom_place_left', 'drawer_bottom_place_right',
            'dice1', 'dice2',
            'trash', 'trash_bin'
        ]

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that some objects may be referenced in code but not present in the object list.
        # We will check for this and only act on objects that are present in the object list.

        # Example: Suppose the plan wants to pick 'dice1' and 'dice2' and place them in a drawer.
        # We will check if these objects exist before acting.

        # Define the objects to manipulate (from the plan/problem)
        objects_to_pick = ['dice1', 'dice2']
        drawer_name = 'drawer_top'
        drawer_handle = 'drawer_top_handle'
        drawer_location = 'drawer_top_place_left'  # Example location for the drawer
        ready_pose = 'ready-pose'

        # Check for missing objects
        for obj in objects_to_pick:
            if obj not in object_list:
                print(f"[Exploration] Object '{obj}' not in object list. Skipping.")
                continue

        # === Main Task Plan Execution ===
        # The plan: For each dice, pick it up and place it in the drawer.
        # The robot must open the drawer first if needed, and return to ready pose at the end.

        # 1. Move to ready pose if not already there
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=positions.get('robot', ready_pose),
                to_location=ready_pose
            )
        except Exception as e:
            print(f"[Error] Failed to move to ready pose: {e}")

        # 2. Open the drawer (if not already open)
        # Pick the handle
        if drawer_handle in object_list:
            try:
                obs, reward, done = execute_pick_handle(
                    env,
                    task,
                    handle=drawer_handle,
                    location=drawer_location
                )
            except Exception as e:
                print(f"[Error] Failed to pick handle '{drawer_handle}': {e}")
        else:
            print(f"[Exploration] Drawer handle '{drawer_handle}' not in object list. Skipping drawer open.")

        # Pull the drawer open
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=drawer_handle,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Error] Failed to pull drawer '{drawer_name}': {e}")

        # 3. For each dice, pick and place in the drawer
        for obj in objects_to_pick:
            if obj not in object_list:
                print(f"[Exploration] Object '{obj}' not in object list. Skipping.")
                continue

            # Pick the object
            try:
                obs, reward, done = execute_pick_object(
                    env,
                    task,
                    obj=obj,
                    location=positions.get(obj, ready_pose)
                )
            except Exception as e:
                print(f"[Error] Failed to pick object '{obj}': {e}")
                continue

            # Place the object in the drawer
            try:
                obs, reward, done = execute_place_object(
                    env,
                    task,
                    obj=obj,
                    drawer=drawer_name,
                    location=drawer_location
                )
            except Exception as e:
                print(f"[Error] Failed to place object '{obj}' in drawer '{drawer_name}': {e}")
                continue

            # After each pick/place, the robot may need to return to ready pose
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=drawer_location
                )
            except Exception as e:
                print(f"[Error] Failed to return to ready pose after placing '{obj}': {e}")

        # 4. Close the drawer
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Error] Failed to push (close) drawer '{drawer_name}': {e}")

        # 5. Return to ready pose at the end
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_location
            )
        except Exception as e:
            print(f"[Error] Failed to return to ready pose at end: {e}")

        print("[Task] Task execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
