# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List Consistency Check (from feedback) ===
        # Ensure all objects used in the plan are present in the object list
        object_names = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2",
            "trash",
            "trash_bin"
        ]
        # Check for missing objects in positions
        for obj in object_names:
            if obj not in positions:
                print(f"[Warning] Object '{obj}' not found in environment positions.")

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check if the drawer is locked before placing objects
        # We'll use exploration skills to determine lock status if needed

        # For demonstration, let's assume we want to put dice1 and dice2 into the middle drawer
        # and need to ensure the drawer is unlocked before proceeding

        # Define relevant names
        dice1 = "dice1"
        dice2 = "dice2"
        drawer = "drawer_middle"
        drawer_handle = "drawer_middle_handle"
        drawer_place = "drawer_middle_place"
        robot_location = "ready-pose"  # Assume starting at ready-pose

        # Step 1: Move to dice1 location and pick it up
        try:
            dice1_pos = positions[dice1]
            obs, reward, done = execute_go(env, task, robot_location, dice1)
            robot_location = dice1
            obs, reward, done = execute_pick(env, task, dice1, dice1)
        except Exception as e:
            print(f"[Error] Failed to pick up {dice1}: {e}")

        # Step 2: Move to ready-pose (if needed)
        try:
            obs, reward, done = execute_go_ready(env, task, robot_location)
            robot_location = "ready-pose"
        except Exception as e:
            print(f"[Error] Failed to return to ready-pose: {e}")

        # Step 3: Move to dice2 location and pick it up
        try:
            dice2_pos = positions[dice2]
            obs, reward, done = execute_go(env, task, robot_location, dice2)
            robot_location = dice2
            obs, reward, done = execute_pick(env, task, dice2, dice2)
        except Exception as e:
            print(f"[Error] Failed to pick up {dice2}: {e}")

        # Step 4: Move to ready-pose (if needed)
        try:
            obs, reward, done = execute_go_ready(env, task, robot_location)
            robot_location = "ready-pose"
        except Exception as e:
            print(f"[Error] Failed to return to ready-pose: {e}")

        # Step 5: Move to drawer handle location to check lock status
        try:
            obs, reward, done = execute_go(env, task, robot_location, drawer_handle)
            robot_location = drawer_handle
        except Exception as e:
            print(f"[Error] Failed to move to drawer handle: {e}")

        # Step 6: Exploration - Try to pull the drawer to check if it's locked
        # If the drawer is locked, this should fail or indicate lock status
        try:
            obs, reward, done = execute_pick(env, task, drawer_handle, drawer_handle)
            obs, reward, done = execute_pull(env, task, drawer, drawer_handle, drawer_handle)
            # If pull succeeds, drawer is unlocked
            drawer_locked = False
        except Exception as e:
            print(f"[Exploration] Drawer may be locked or pull failed: {e}")
            drawer_locked = True

        # Step 7: If drawer is locked, abort or handle accordingly
        if drawer_locked:
            print("[Task] Drawer is locked. Cannot place objects inside.")
        else:
            # Step 8: Place dice1 in the drawer
            try:
                obs, reward, done = execute_place(env, task, dice1, drawer, drawer_place)
            except Exception as e:
                print(f"[Error] Failed to place {dice1} in drawer: {e}")

            # Step 9: Place dice2 in the drawer
            try:
                obs, reward, done = execute_place(env, task, dice2, drawer, drawer_place)
            except Exception as e:
                print(f"[Error] Failed to place {dice2} in drawer: {e}")

            # Step 10: Push the drawer closed
            try:
                obs, reward, done = execute_push(env, task, drawer, drawer_place)
            except Exception as e:
                print(f"[Error] Failed to push drawer closed: {e}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
