# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to ensure that dice are recognized as physical objects.
        # We'll perform exploration to check for missing predicates (e.g., physical-object, identified, etc.)
        # This phase is to ensure the environment and objects are correctly set up for manipulation.

        # List of dice and handles (physical objects)
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_names = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # Step 1: Move robot to each die and attempt to identify it (simulate exploration)
        for die in dice_names:
            try:
                die_pos = positions.get(die, None)
                if die_pos is None:
                    print(f"[Exploration] Warning: Position for {die} not found.")
                    continue
                # Move robot to die's location (simulate identification)
                print(f"[Exploration] Moving to {die} at {die_pos} to identify.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='ready-pose',
                    to_location=die_pos,
                    max_steps=100
                )
                # Optionally, mark as identified (if such a predicate exists in the environment)
                # In this code, we just simulate the exploration step.
            except Exception as e:
                print(f"[Exploration] Exception during identification of {die}: {e}")

        # Step 2: Check if dice are physical objects (simulate by attempting to pick)
        for die in dice_names:
            try:
                die_pos = positions.get(die, None)
                if die_pos is None:
                    print(f"[Exploration] Warning: Position for {die} not found.")
                    continue
                print(f"[Exploration] Attempting to pick {die} at {die_pos}.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=die,
                    location=die_pos,
                    max_steps=100
                )
                if done:
                    print(f"[Exploration] Successfully picked {die}.")
                else:
                    print(f"[Exploration] Could not pick {die}.")
            except Exception as e:
                print(f"[Exploration] Exception during pick of {die}: {e}")

        # === Main Task Phase: Insert one physical die into a drawer ===
        # For demonstration, insert dice1 into drawer_bottom

        die_to_insert = 'dice1'
        drawer_to_use = 'drawer_bottom'
        handle_to_use = 'handle_bottom'
        die_pos = positions.get(die_to_insert, None)
        handle_pos = positions.get(handle_to_use, None)
        drawer_pos = positions.get(drawer_to_use, None)

        # 1. Move to die and pick it up
        try:
            print(f"[Task] Moving to {die_to_insert} at {die_pos} to pick.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=die_pos,
                max_steps=100
            )
            print(f"[Task] Picking up {die_to_insert}.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=die_to_insert,
                location=die_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during pick of {die_to_insert}: {e}")

        # 2. Move to ready-pose (if needed)
        try:
            print(f"[Task] Returning to ready-pose.")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=die_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during go_ready: {e}")

        # 3. Move to handle and pick handle
        try:
            print(f"[Task] Moving to {handle_to_use} at {handle_pos} to pick handle.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=handle_pos,
                max_steps=100
            )
            print(f"[Task] Picking up handle {handle_to_use}.")
            obs, reward, done = execute_pick_handle(
                env,
                task,
                handle_name=handle_to_use,
                location=handle_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during pick of handle {handle_to_use}: {e}")

        # 4. Pull the drawer open
        try:
            print(f"[Task] Pulling drawer {drawer_to_use} using handle {handle_to_use}.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_to_use,
                handle_name=handle_to_use,
                location=handle_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during pull of {drawer_to_use}: {e}")

        # 5. Move to ready-pose (if needed)
        try:
            print(f"[Task] Returning to ready-pose after pull.")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=handle_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during go_ready after pull: {e}")

        # 6. Move to drawer and place die inside
        try:
            print(f"[Task] Moving to {drawer_to_use} at {drawer_pos} to place die.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=drawer_pos,
                max_steps=100
            )
            print(f"[Task] Placing {die_to_insert} into {drawer_to_use}.")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=die_to_insert,
                drawer_name=drawer_to_use,
                location=drawer_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during place of {die_to_insert} into {drawer_to_use}: {e}")

        # 7. Move to ready-pose (if needed)
        try:
            print(f"[Task] Returning to ready-pose after place.")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during go_ready after place: {e}")

        # 8. Push the drawer closed
        try:
            print(f"[Task] Moving to {drawer_to_use} at {drawer_pos} to push drawer closed.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=drawer_pos,
                max_steps=100
            )
            print(f"[Task] Pushing drawer {drawer_to_use} closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_to_use,
                location=drawer_pos,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Exception during push of {drawer_to_use}: {e}")

        print("[Task] Physical die insertion task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()