# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use the predefined primitives; do not redefine any skill

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, extended for exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()

        # --- BEGIN EXPLORATION PHASE FOR MISSING PREDICATE ---
        print("[Exploration] Attempting to discover cause of blocked actions...")

        # Step 1: Try to execute an action (e.g., execute_pick) and catch PredicateError
        #         If action fails, try to find out which predicate is missing/in effect
        # For demonstration, suppose we attempt to perform 'execute_pick' and examine if (need-ready) is blocking

        exploration_success = False
        exploration_attempted = False

        # For exploration: check robot at ready pose, find a pickable object, location info
        # These would normally come from the observation or positions.
        # The keys may differ depending on scenario; handle missing info gracefully
        
        robot_location = None
        pickable_obj = None
        pickable_obj_loc = None

        for obj, pos in positions.items():
            if obj.startswith("robot") or obj.startswith("Robot") or obj == "robot":
                robot_location = pos
            # Suppose: look for some object on the floor to pick
            if "floor" in obj or "obj" in obj or "object" in obj:
                pickable_obj = obj
                pickable_obj_loc = pos
                break

        # If we don't have detailed info, set defaults (these may fail, which is partly intended for exploration)
        if not robot_location:
            robot_location = (0.0, 0.0, 0.0)
        if not pickable_obj:
            # Try a default key or give up
            pickable_obj = list(positions.keys())[0] if positions else "object_1"
            pickable_obj_loc = positions.get(pickable_obj, (0.1, 0.0, 0.0))

        print(f"[Exploration] Robot location: {robot_location}, Object: {pickable_obj}, Object location: {pickable_obj_loc}")

        try:
            # Try 'execute_pick' (if in available skills)
            if "execute_pick" in globals():
                print("[Exploration] Attempting execute_pick...")
                # The parameters depend on the actual skill signature; adapting to likely input
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target=pickable_obj,
                    location=pickable_obj_loc,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_success = True
            else:
                print("[Exploration] 'execute_pick' skill not available, skipping direct pick attempt.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_pick: {e}")
            # Examine feedback / context
            # If error message mentions 'need-ready', it's likely that predicate is the blocker.
            if 'need-ready' in str(e).lower():
                print("[Exploration] Detected that (need-ready) predicate is missing/active and blocking skill execution.")
                exploration_attempted = True
            else:
                print("[Exploration] Unknown error, possibly unrelated to (need-ready):", e)
            exploration_success = False

        # If explore revealed 'need-ready', try to reset it using available skill
        if exploration_attempted or not exploration_success:
            # Try to execute the 'execute_go_ready' skill if available
            if "execute_go_ready" in globals():
                try:
                    print("[Exploration] Attempting to clear (need-ready) by running execute_go_ready...")
                    # The parameter is source location, which we set using robot_location if known
                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        from_location=robot_location,
                        approach_distance=0.15,
                        max_steps=80,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=8.0,
                    )
                    print("[Exploration] (need-ready) cleared by execute_go_ready skill.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go_ready: {e}")
            else:
                print("[Exploration] 'execute_go_ready' not available in skill set, cannot clear (need-ready).")
        else:
            print("[Exploration] No need to clear (need-ready) or action succeeded.")

        print("[Exploration] Missing predicate check for (need-ready) exploration phase complete.")
        # --- END EXPLORATION PHASE ---

        # === Continue with domain plan/skills as normal ===
        # Insert your oracle plan execution code here using the available skills
        # For demo, just print that exploration succeeded and ready to proceed
        print("[Task] Proceeding to main oracle plan now (not implemented here).")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
