# run_skeleton_task.py (Completed - with Exploration Logic for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only these predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicate.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their positions
        positions = get_object_positions()

        # ---------------- EXPLORATION PHASE: Identify Missing Predicate ----------------
        # Feedback: (drawer-open drawer-bottom) -- implies that this literal is missing
        # To explore, we will check if drawer-bottom is open, or needs to be opened, and which predicate is missing.

        # Identify the relevant objects
        drawer_name = 'drawer-bottom'
        handle_name = None
        robot_location = None

        # Try to find the handle for the drawer (search for is-handle/handle-of per domain)
        for name in positions.keys():
            if 'handle' in name and 'bottom' in name:
                handle_name = name
                break
        if handle_name is None:
            # Fallback: get any handle for bottom drawer
            for name in positions.keys():
                if 'handle' in name:
                    handle_name = name
                    break

        # Find robot's start location
        for name in positions.keys():
            if 'robot' in name:
                robot_location = name
                break
        if robot_location is None:
            robot_location = 'ready-pose'  # fallback default

        # Get start position for planning
        try:
            robot_pos = positions[robot_location]
        except Exception:
            robot_pos = [0,0,0]

        # Retrieve positions for objects (could be unused, but left for completeness)
        try:
            drawer_pos = positions[drawer_name]
        except Exception:
            drawer_pos = [0,0,0]
        try:
            handle_pos = positions[handle_name] if handle_name is not None else None
        except Exception:
            handle_pos = None

        # PHASE 1: Move to drawer-bottom's vicinity if needed
        print("[Exploration] Moving robot to drawer location if not already there.")
        try:
            # We don't know current robot location; environment may use only logical locations.
            # Assume a logical skill for this domain: execute_go(from, to)
            # Try to move robot to the drawer's location, skipping if domains don't match.
            obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=drawer_name)
            print(f"Moved to {drawer_name}")
        except Exception as e:
            print(f"[Warning] Could not move robot to drawer location: {str(e)}")

        # PHASE 2: Try to open the drawer to check the missing predicate
        print("[Exploration] Attempting to pull (open) the drawer-bottom with its handle.")
        try:
            # Pick the handle first (if not already holding)
            if handle_name is not None:
                # Try picking up the handle: execute_pick_handle(handle, location)
                obs, reward, done = execute_pick_handle(env, task, h=handle_name, p=drawer_name)
                print(f"Picked handle {handle_name} at {drawer_name}")

            # Now try pulling: execute_pull(drawer, handle, location)
            obs, reward, done = execute_pull(env, task, d=drawer_name, h=handle_name, p=drawer_name)
            print(f"Pulled/opened {drawer_name} using {handle_name}")
        except Exception as e:
            print(f"[Warning] Could not open/pull the drawer: {str(e)}")

        # PHASE 3: Exploration result: Report observed robot state and drawer state
        # Here, check if the predicate (drawer-open drawer-bottom) is now true/observable.
        # In practice, you would query predicates/state; here, log the step and inform user.
        print("[Exploration] Exploration complete. If failures occurred above, likely missing predicate is (drawer-open drawer-bottom).")
        print("[Exploration] Please check if your planner or state observer is managing this predicate.")

        # ---------- (Optional) Continue with Task Plan ----------
        # If the task is to put something in the drawer, or check its state, proceed accordingly.
        # Since this is an exploration phase, we end here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
