# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Skill functions provided externally

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation (optional)
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()

        # Exploration phase: determine missing predicate by triggering exploration skills
        print("[Exploration] Starting predicate discovery phase.")
        missing_predicate_found = False

        # Example assumptions for this exploration: 
        # - There are objects and robot/agent location(s)
        # - We attempt each exploration action to see which predicates are missing/accessed
        # - Skill names (from available skills): ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #   Only these can be executed. We may not have direct access to e.g. execute_go_identify, so focus on primitives.

        # Find a location and an object to interact with:
        robot_location = None
        obj_to_explore = None
        drawer_to_explore = None
        handle_to_explore = None
        for name, pos in positions.items():
            if 'robot' in name or 'robby' in name:
                robot_location = name
            if 'drawer' in name and drawer_to_explore is None:
                drawer_to_explore = name
            if ('handle' in name or 'Handle' in name) and handle_to_explore is None:
                handle_to_explore = name
            elif obj_to_explore is None and 'object' in name:
                obj_to_explore = name
        # Fallbacks in case of non-informative names
        position_keys = list(positions.keys())
        if not robot_location and len(position_keys) > 0:
            robot_location = position_keys[0]
        if not obj_to_explore and len(position_keys) > 1:
            obj_to_explore = position_keys[1]

        # Exploration Steps
        # 1. Try 'execute_pick' on an object to see if precondition fails (missing predicate?).
        print(f"[Exploration] Attempting to pick up: {obj_to_explore} at {robot_location} (simulate on-floor).")
        try:
            obs, reward, done, info = execute_pick(env, task, obj_to_explore, robot_location)
            print("[Exploration] execute_pick succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")
            missing_predicate_found = True

        # 2. Try 'execute_pull' to check for lock-known or lock predicates involved.
        if handle_to_explore and drawer_to_explore:
            print(f"[Exploration] Attempting to pull handle: {handle_to_explore} on drawer: {drawer_to_explore} at {robot_location}.")
            try:
                obs, reward, done, info = execute_pull(env, task, drawer_to_explore, handle_to_explore, robot_location)
                print("[Exploration] execute_pull succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")
                missing_predicate_found = True

        # 3. Try 'execute_push' if drawer info exists.
        if drawer_to_explore:
            print(f"[Exploration] Attempting to push drawer: {drawer_to_explore} at {robot_location}.")
            try:
                obs, reward, done, info = execute_push(env, task, drawer_to_explore, robot_location)
                print("[Exploration] execute_push succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")
                missing_predicate_found = True

        # 4. Try 'execute_go' to another location, if available.
        another_location = None
        for k in positions.keys():
            if k != robot_location:
                another_location = k
                break
        if another_location:
            print(f"[Exploration] Attempting to move from {robot_location} to {another_location}.")
            try:
                obs, reward, done, info = execute_go(env, task, robot_location, another_location)
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")
                missing_predicate_found = True

        # Report exploration result
        if missing_predicate_found:
            print("[Exploration] There may be a missing predicate in the domain/problem definition.")
        else:
            print("[Exploration] No missing predicate found through available primitive actions.")

        # Continue with the oracle plan as needed, using the available skills step-by-step
        # For demonstration, you would replace the following with your actual plan:
        # Example sequence:
        #   1. Move robot to object location (if not already there)
        #   2. Pick up object
        #   3. Move to target location (e.g., near a drawer)
        #   4. Place object (or perform other actions as needed)
        # Below is a generic placeholder - replace "obj_to_explore", "robot_location", etc., as necessary based on your task

        print("[Task] Starting oracle plan execution (demo sequence).")

        try:
            # Example: Move to object's location (skip if already there)
            if robot_location != obj_to_explore:
                obs, reward, done, info = execute_go(env, task, robot_location, obj_to_explore)
                robot_location = obj_to_explore
                print(f"[Task] Robot moved to {robot_location}.")
            # Pick up the object
            obs, reward, done, info = execute_pick(env, task, obj_to_explore, robot_location)
            print(f"[Task] Picked up object {obj_to_explore}.")
            # Move to drawer location (if not at same position)
            if robot_location != drawer_to_explore:
                obs, reward, done, info = execute_go(env, task, robot_location, drawer_to_explore)
                robot_location = drawer_to_explore
                print(f"[Task] Robot moved to {robot_location} (drawer).")
            # Place the object in the drawer
            if drawer_to_explore:
                obs, reward, done, info = execute_place(env, task, obj_to_explore, drawer_to_explore, robot_location)
                print(f"[Task] Placed object in drawer {drawer_to_explore}")
            print("[Task] Oracle plan execution completed.")

        except Exception as e:
            print(f"[Task] Action execution failed: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()