# run_skeleton_task.py (Task Execution with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills (execute_pick, execute_place, execute_push, execute_pull, etc.)

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and State Information ===
        # Example: positions = {'obj_a': (x, y, z), 'handle': (x, y, z), 'drawer': (x, y, z), ...}
        positions = get_object_positions()
        
        # Extract main objects for demonstration/exploration
        # Object names may need to be updated according to your specific environment
        try:
            # Try to guess main entities (update these keys as necessary for your problem)
            # These are only examples; change as appropriate.
            object_keys = list(positions.keys())
            obj_candidate = None
            handle_candidate = None
            drawer_candidate = None
            location_keys = set()
            for k in object_keys:
                if "handle" in k:
                    handle_candidate = k
                elif "drawer" in k:
                    drawer_candidate = k
                elif "object" in k or "obj" in k or "item" in k:
                    obj_candidate = k
                elif "location" in k or "pose" in k or "room" in k:
                    location_keys.add(k)
            if not obj_candidate:
                obj_candidate = object_keys[0] if object_keys else None
            # Fallback
            if not drawer_candidate:
                for k in object_keys:
                    if "drawer" in k:
                        drawer_candidate = k
                        break
            # Fallback to any available
            if not handle_candidate:
                for k in object_keys:
                    if "handle" in k:
                        handle_candidate = k
            # Use arbitrary locations as source and goal
            location_list = list(location_keys) if location_keys else []
        except Exception as e:
            print("[Error] Could not parse object positions:", e)
            obj_candidate = None
            handle_candidate = None
            drawer_candidate = None
            location_list = []

        # Exploration Phase: Try available skills in contexts to elicit errors or missing predicates
        print("[Exploration] Testing available skills for predicate discovery...")
        missing_predicates = set()
        skills_to_try = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Prepare dummy or discovered names
        robot_location_key = None
        for k in object_keys:
            if 'robot' in k and 'at' in k:
                robot_location_key = k
                break
        # Fallback: guess from observation (as we can't define robot names from this code)
        current_robot_location = None
        if hasattr(task, 'get_robot_location'):
            current_robot_location = task.get_robot_location()
        elif 'robot_location' in positions:
            current_robot_location = positions['robot_location']
        else:
            # Just use first available location if any
            current_robot_location = location_list[0] if location_list else None

        # Try to call each skill with plausible or dummy arguments
        # You may need to manually adapt these names per your setup!
        for skill_name in skills_to_try:
            try:
                print(f"[Exploration] Trying skill: {skill_name}")
                skill_fn = globals().get(skill_name)
                if not skill_fn:
                    print(f"  [Warning] Skill function {skill_name} not found in skill_code.")
                    continue
                if skill_name == "execute_pick":
                    # Try to pick main object at robot's current location
                    res = skill_fn(env, task, obj_candidate, current_robot_location)
                elif skill_name == "execute_place":
                    # Try to place object into drawer at robot's current location
                    res = skill_fn(env, task, obj_candidate, drawer_candidate, current_robot_location)
                elif skill_name == "execute_push":
                    # Try to push the drawer at current location
                    res = skill_fn(env, task, drawer_candidate, current_robot_location)
                elif skill_name == "execute_pull":
                    # Try to pull the drawer with a handle at current location
                    res = skill_fn(env, task, drawer_candidate, handle_candidate, current_robot_location)
                elif skill_name == "execute_sweep":
                    # Sweep an object at the location
                    res = skill_fn(env, task, obj_candidate, current_robot_location)
                elif skill_name == "execute_rotate":
                    # Test rotate with the main object or drawer
                    res = skill_fn(env, task, obj_candidate or drawer_candidate)
                elif skill_name == "execute_go":
                    # Move from one location to another (try between two known locations)
                    if len(location_list) >= 2:
                        from_loc = location_list[0]
                        to_loc = location_list[1]
                        res = skill_fn(env, task, from_loc, to_loc)
                    else:
                        print("  [Exploration] Not enough locations known for execute_go.")
                        continue
                elif skill_name == "execute_gripper":
                    # Gripper action may take no or default arguments
                    res = skill_fn(env, task)
                else:
                    print(f"  [Exploration] Skill {skill_name} signature unknown.")
                    continue
            except Exception as ex:
                # Collect feedback about missing predicates, preconditions, or types
                error_msg = str(ex)
                print(f"  [Exploration] Exception while calling {skill_name}: {error_msg}")
                if "predicate" in error_msg or "precondition" in error_msg or "missing" in error_msg:
                    missing_predicates.add(error_msg)
                continue

        if missing_predicates:
            print("[Exploration] Possible missing predicates identified from skill execution:")
            for pred in missing_predicates:
                print(f"   - {pred}")
        else:
            print("[Exploration] No explicit predicate errors identified from skill execution.")

        # === (Placeholder) — Here you would insert the actual oracle plan as skill calls ===
        # For demonstration, provide a mockup of task plan execution using available skills
        # NOTE: Replace these with the real plan steps for your task/goal

        print("[Task] Executing ORACLE PLAN (mockup example, replace with your plan):")
        try:
            # Example skill execution sequence
            # 1. Move robot to source location (assuming location_list[0]->location_list[1] exists)
            if len(location_list) >= 2:
                obs, reward, done = execute_go(env, task, location_list[0], location_list[1])
                print(f"  [Plan] execute_go({location_list[0]} -> {location_list[1]})")

            # 2. Pick the object at destination
            if obj_candidate and location_list:
                obs, reward, done = execute_pick(env, task, obj_candidate, location_list[1])
                print(f"  [Plan] execute_pick({obj_candidate}, {location_list[1]})")

            # 3. Place the object into drawer at the same location
            if drawer_candidate and obj_candidate and location_list:
                obs, reward, done = execute_place(env, task, obj_candidate, drawer_candidate, location_list[1])
                print(f"  [Plan] execute_place({obj_candidate}, {drawer_candidate}, {location_list[1]})")

            # 4. Optionally, pull the drawer if needed
            if drawer_candidate and handle_candidate and location_list:
                obs, reward, done = execute_pull(env, task, drawer_candidate, handle_candidate, location_list[1])
                print(f"  [Plan] execute_pull({drawer_candidate}, {handle_candidate}, {location_list[1]})")

            # 5. Optionally, push the drawer
            if drawer_candidate and location_list:
                obs, reward, done = execute_push(env, task, drawer_candidate, location_list[1])
                print(f"  [Plan] execute_push({drawer_candidate}, {location_list[1]})")

            print("[Task] Plan execution completed.")

        except Exception as plan_ex:
            print(f"[Error] Exception during task plan execution: {plan_ex}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()