# run_skeleton_task.py (Completed for Exploration of (need-ready) Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # PREDEFINED: do not redefine or modify skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with an exploration phase focused on missing predicate detection (need-ready).'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Investigate 'need-ready' predicate ===
        # According to feedback, there is a control or precondition involving (need-ready).
        # We explore the environment using only the available skills to determine when 'need-ready' blocks skills,
        # and when it is required to return to ready-pose.

        # Determine sample objects and locations from the positions dictionary
        objects = [k for k in positions if not k.startswith('drawer') and not k.startswith('handle')]
        drawers = [k for k in positions if k.startswith('drawer')]
        handles = [k for k in positions if k.startswith('handle')]
        locations = set()
        for pos in positions:
            loc = positions[pos]
            if isinstance(loc, (list, tuple)):
                locations.add(tuple(loc))
        locations = list(locations)
        # Fallback if positions dict is not detailed enough
        if len(locations) == 0 and len(positions) > 0:
            # Take as single dummy location
            locations = [positions[next(iter(positions))]]

        # Try to find an object and a drawer and a handle for exploration
        test_obj = objects[0] if objects else None
        test_drawer = drawers[0] if drawers else None
        test_handle = handles[0] if handles else None
        test_loc = positions[test_obj] if test_obj else locations[0] if locations else None

        # Initial exploration sequence
        # 1. Attempt to pick an object ("execute_pick") to potentially set need-ready.
        # 2. Try to immediately use another skill and see if blocked.
        # 3. Use "execute_go_ready" to clear need-ready.
        # 4. Continue with further actions.
        # This checks what skills are blocked when need-ready is set.

        print("[Exploration] Attempting to pick object:", test_obj, "at", test_loc)
        pick_success = False
        try:
            obs, reward, done = execute_pick(env, task, obj=test_obj, location=test_loc)
            pick_success = True
            print("[Exploration] execute_pick succeeded.")
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)

        # Try to execute another skill that should be blocked by need-ready (e.g., a move/go)
        move_success = False
        if pick_success:
            print("[Exploration] Attempting to execute_go after execute_pick to see if blocked by need-ready.")
            try:
                # Find a different location to go
                other_locations = [l for l in locations if l != test_loc]
                if other_locations:
                    obs, reward, done = execute_go(env, task, from_location=test_loc, to_location=other_locations[0])
                else:
                    # Try going to the same location (no effect)
                    obs, reward, done = execute_go(env, task, from_location=test_loc, to_location=test_loc)
                move_success = True
                print("[Exploration] execute_go succeeded (NOT blocked by need-ready).")
            except Exception as e:
                print("[Exploration] execute_go failed (possibly blocked by need-ready):", e)

        # If execute_go failed due to need-ready, now attempt to execute_go_ready
        if pick_success and not move_success:
            print("[Exploration] Attempting to execute_go_ready to clear need-ready state.")
            try:
                obs, reward, done = execute_go_ready(env, task, from_location=test_loc)
                print("[Exploration] execute_go_ready succeeded. Need-ready cleared.")
                # Now try to move again
                try:
                    other_locations = [l for l in locations if l != test_loc]
                    if other_locations:
                        obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location=other_locations[0])
                    else:
                        obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location='ready-pose')
                    print("[Exploration] execute_go after execute_go_ready succeeded.")
                except Exception as e:
                    print("[Exploration] execute_go after execute_go_ready failed:", e)
            except Exception as e:
                print("[Exploration] execute_go_ready failed:", e)

        # For further exploration: Try to pick a handle (should not set need-ready), then move
        if test_handle:
            handle_loc = positions[test_handle]
            print("[Exploration] Attempting to pick handle:", test_handle, "at", handle_loc)
            try:
                obs, reward, done = execute_pick(env, task, obj=test_handle, location=handle_loc)
                print("[Exploration] execute_pick (handle) succeeded. Now execute_go.")
                # Now test movement right after picking handle (should NOT require ready)
                other_locations = [l for l in locations if l != handle_loc]
                goto = other_locations[0] if other_locations else handle_loc
                try:
                    obs, reward, done = execute_go(env, task, from_location=handle_loc, to_location=goto)
                    print("[Exploration] After picking handle, execute_go succeeded (as expected, no need-ready).")
                except Exception as e:
                    print("[Exploration] After picking handle, execute_go failed:", e)
            except Exception as e:
                print("[Exploration] execute_pick (handle) failed:", e)

        print("[Exploration] Exploration phase finished. Please check logs for state transitions involving 'need-ready'.")

        # === End of Exploration ===

        # After gathering these transitions, you would use this knowledge to plan next steps
        # (i.e., you must insert execute_go_ready after actions that set need-ready, before blocked skills).
        # For this code, we end at exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()