# run_skeleton_task.py (Completed, with Exploration Phase Using Only Available Skills)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use existing skill functions only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with an exploration phase for missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # e.g., {'drawer1': (x1,y1,z1), 'handle1': (x2,y2,z2), ...}

        # ------------------------------------------------
        # 1. EXPLORATION PHASE: Discover missing predicates
        # ------------------------------------------------

        # We know from feedback that (drawer-unlocked drawer1) is missing or not true/unknown.
        # Our goal is to discover/push the system to realize whether the drawer is locked/unlocked.
        # In the exploration domain, execute_pull (with required args) will set (lock-known ?obj).

        # Try to find a handle and the associated drawer
        drawer_name = None
        handle_name = None
        for obj in positions.keys():
            if "handle" in obj:
                handle_name = obj
                # Try to find associated drawer by convention
                suffix = ''.join(filter(str.isdigit, obj))
                candidate = f"drawer{suffix}"
                if candidate in positions:
                    drawer_name = candidate
                    break
        # If not found, fallback to any drawer
        if not drawer_name:
            for obj in positions.keys():
                if "drawer" in obj:
                    drawer_name = obj
                    break
        if not handle_name:
            for obj in positions.keys():
                if "handle" in obj:
                    handle_name = obj
                    break

        # Get handle/drawer positions for use in skill calls
        drawer_pos = positions.get(drawer_name, None)
        handle_pos = positions.get(handle_name, None)

        # Get current robot position if available
        robot_pos = None
        for obj, pos in positions.items():
            if "robot" in obj or "ready" in obj:
                robot_pos = pos
                break

        # --- Step 1: Move to handle location (if needed) ---
        print("[Exploration] Moving to handle to test drawer lock state...")
        try:
            if robot_pos and handle_pos and robot_pos != handle_pos:
                obs, reward, done = execute_go(
                    env, task,
                    from_pos=robot_pos,
                    to_pos=handle_pos,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                robot_pos = handle_pos
        except Exception as e:
            print("[Warn] Failed to move to handle:", e)

        # --- Step 2: Try to pick the handle (if not holding) ---
        print("[Exploration] Attempting to pick the handle...")
        try:
            obs, reward, done = execute_pick(
                env, task,
                target=handle_name,
                at_location=handle_pos,
                max_steps=100,
                approach_distance=0.10,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print("[Warn] Failed to pick handle:", e)

        # --- Step 3: Attempt to pull the drawer using handle
        # This should help the system determine if (drawer-unlocked drawer1) is true
        print("[Exploration] Attempting to pull the drawer (probe lock state)...")
        try:
            obs, reward, done = execute_pull(
                env, task,
                drawer=drawer_name,
                handle=handle_name,
                at_location=handle_pos,
                max_steps=100,
                timeout=10.0
            )
            # If action succeeded, we have now possibly established (drawer-unlocked drawer1).
            print("[Info] Pull executed, lock-known state may be updated.")
        except Exception as e:
            print("[Warn] Failed to pull the drawer:", e)

        # ----------- End of exploration phase --------------
        # At this point, either the missing predicate is discovered
        # or we have confirmed the lock state of the drawer.

        # ------------------------------------------------
        # 2. TASK PLAN PHASE: Use Real Domain Skills
        # ------------------------------------------------

        # Example canonical scenario:
        # - Pick up an object and place it in the (possibly unlocked/open) drawer.

        # Find a floor object that is NOT a handle
        obj_to_manipulate = None
        obj_pos = None
        for obj, pos in positions.items():
            if "ball" in obj or ("obj" in obj and "handle" not in obj):
                obj_to_manipulate = obj
                obj_pos = pos
                break
        # Fallback: just pick any object that is not a handle/drawer
        if not obj_to_manipulate:
            for obj in positions:
                if "handle" not in obj and "drawer" not in obj and "robot" not in obj and "ready" not in obj:
                    obj_to_manipulate = obj
                    obj_pos = positions[obj]
                    break

        # Move to object if not already there
        if obj_to_manipulate and obj_pos:
            print(f"[Task] Moving to object: {obj_to_manipulate} at {obj_pos}")
            if robot_pos != obj_pos:
                try:
                    obs, reward, done = execute_go(
                        env, task,
                        from_pos=robot_pos,
                        to_pos=obj_pos,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    robot_pos = obj_pos
                except Exception as e:
                    print("[Warn] Could not move to object:", e)

            # Pick the object
            print(f"[Task] Picking object: {obj_to_manipulate}")
            try:
                obs, reward, done = execute_pick(
                    env, task,
                    target=obj_to_manipulate,
                    at_location=obj_pos,
                    max_steps=100,
                    approach_distance=0.10,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Warn] Pick failed for {obj_to_manipulate}:", e)

            # Move to drawer location
            print(f"[Task] Moving to drawer at {drawer_pos}")
            try:
                obs, reward, done = execute_go(
                    env, task,
                    from_pos=obj_pos,
                    to_pos=drawer_pos,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                robot_pos = drawer_pos
            except Exception as e:
                print("[Warn] Move to drawer failed:", e)

            # Place the object in the drawer if open
            print(f"[Task] Placing object: {obj_to_manipulate} in drawer: {drawer_name}")
            try:
                obs, reward, done = execute_place(
                    env, task,
                    target=obj_to_manipulate,
                    drawer=drawer_name,
                    at_location=drawer_pos,
                    max_steps=100,
                    approach_distance=0.10,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Task] Place completed.")
            except Exception as e:
                print("[Warn] Place failed:", e)

            # Optionally, push drawer to close
            print(f"[Task] Pushing the drawer to close it.")
            try:
                obs, reward, done = execute_push(
                    env, task,
                    drawer=drawer_name,
                    at_location=drawer_pos,
                    max_steps=100,
                    timeout=10.0
                )
                print("[Task] Drawer push/close completed.")
            except Exception as e:
                print("[Warn] Push failed:", e)
        else:
            print("[Error] No suitable object found to manipulate.")

        print("===== Task execution completed =====")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()