# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate Identification)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the skills provided externally.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including required exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Dictionary: e.g., positions = {'obj1': (x1,y1,z1), 'drawer1': (x2,y2,z2), ...}
        positions = get_object_positions()

        # === Exploration phase for missing predicate ===

        # The planning feedback indicated that the planner timed out: it likely failed because some required knowledge (predicate) was "missing".
        # Based on "exploration knowledge", we use available skills to actively attempt knowledge-gathering primitive actions.

        # The skills available include: 
        # 'execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'

        # If possible, try each exploration skill in the beginning and observe the result.
        # (Assume object and location names; adapt as appropriate for your setup.)

        # We'll try to execute each available skill in a safe fashion.
        exploration_done = False
        found_missing_predicate = None
        exploration_candidates = [
            "execute_sweep",
            "execute_gripper",
            "execute_push",
            "execute_pull",
            "execute_pick",
            "execute_place",
            "execute_rotate"
        ]
        
        # We attempt each exploratory action safely and log observations/results.
        for skill in exploration_candidates:
            try:
                if skill == "execute_pick":
                    # Try to pick an object on the floor that's not a handle
                    for name, pos in positions.items():
                        if 'handle' not in name and 'drawer' not in name:
                            print(f"[EXPLORE] Trying execute_pick on {name} at {pos}")
                            obs, reward, done = execute_pick(
                                env,
                                task,
                                target_obj=name,
                                target_pos=pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            if done:
                                print(f"[EXPLORE] execute_pick successful for {name}")
                                found_missing_predicate = "holding"
                                exploration_done = True
                            break
                elif skill == "execute_pull":
                    # Try to pull a handle (if any) -- this could reveal lock-known or state predicates
                    for name, pos in positions.items():
                        if "handle" in name:
                            print(f"[EXPLORE] Trying execute_pull on {name} at {pos}")
                            obs, reward, done = execute_pull(
                                env,
                                task,
                                target_handle=name,
                                target_pos=pos,
                                max_steps=100,
                                threshold=0.01,
                                approach_distance=0.15,
                                timeout=10.0
                            )
                            if done:
                                print(f"[EXPLORE] execute_pull successful for {name}")
                                found_missing_predicate = "drawer-unlocked/drawer-locked or lock-known"
                                exploration_done = True
                            break
                elif skill == "execute_push":
                    # Attempt to push a drawer (if any exist)
                    for name, pos in positions.items():
                        if "drawer" in name:
                            print(f"[EXPLORE] Trying execute_push on {name} at {pos}")
                            obs, reward, done = execute_push(
                                env,
                                task,
                                target_drawer=name,
                                target_pos=pos,
                                max_steps=100,
                                threshold=0.01,
                                approach_distance=0.15,
                                timeout=10.0
                            )
                            if done:
                                print(f"[EXPLORE] execute_push successful for {name}")
                                found_missing_predicate = "drawer-open/closed"
                                exploration_done = True
                            break
                elif skill == "execute_sweep":
                    # Attempt to sweep any object on floor (non-handle)
                    for name, pos in positions.items():
                        if 'handle' not in name and 'drawer' not in name:
                            print(f"[EXPLORE] Trying execute_sweep on {name} at {pos}")
                            obs, reward, done = execute_sweep(
                                env,
                                task,
                                target_obj=name,
                                target_pos=pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            if done:
                                print(f"[EXPLORE] execute_sweep successful for {name}")
                                found_missing_predicate = "on-floor"
                                exploration_done = True
                            break
                elif skill == "execute_gripper":
                    # Just try gripper ancillary skill if present
                    print(f"[EXPLORE] Trying execute_gripper")
                    obs, reward, done = execute_gripper(
                        env,
                        task
                    )
                    print(f"[EXPLORE] execute_gripper completed.")
                elif skill == "execute_rotate":
                    # Some robots allow rotating objects in hand or handles
                    for name, pos in positions.items():
                        if 'handle' in name:
                            print(f"[EXPLORE] Trying execute_rotate on {name} at {pos}")
                            obs, reward, done = execute_rotate(
                                env,
                                task,
                                target_handle=name,
                                target_pos=pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            print(f"[EXPLORE] execute_rotate attempted on {name}")
                            break
                elif skill == "execute_place":
                    # Try to place an object in a drawer, if holding
                    for obj_name, pos in positions.items():
                        if 'handle' not in obj_name and 'drawer' not in obj_name:
                            for drawer_name, drawer_pos in positions.items():
                                if 'drawer' in drawer_name:
                                    print(f"[EXPLORE] Trying execute_place of {obj_name} into {drawer_name}")
                                    obs, reward, done = execute_place(
                                        env,
                                        task,
                                        target_obj=obj_name,
                                        target_drawer=drawer_name,
                                        target_pos=drawer_pos,
                                        approach_distance=0.15,
                                        max_steps=100,
                                        threshold=0.01,
                                        approach_axis='z',
                                        timeout=10.0
                                    )
                                    print(f"[EXPLORE] execute_place attempted for {obj_name}")
                                    break
                            break
                if exploration_done:
                    print(f"[EXPLORE] Identified missing predicate candidate: {found_missing_predicate}")
                    break
            except Exception as ex:
                print(f"[EXPLORE] Skill {skill} failed: {str(ex)}")
        if not exploration_done:
            print("[EXPLORE] No definitive missing predicate identified; action exploration complete.")

        # === Main Oracle Plan Execution Block ===
        # Insert your concrete, task-specific plan here using the known skill functions.
        # This would typically appear after the exploration block, following results from exploration.

        # Example for picking and placing, assuming obj and drawer are determined and available.
        # Replace 'obj1', 'drawer1' etc., appropriately.

        try:
            for obj_name, obj_pos in positions.items():
                if 'handle' not in obj_name and 'drawer' not in obj_name:
                    print(f"[TASK] Moving to {obj_name} at {obj_pos}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        source_pos=None,        # Obtain current position from obs if needed
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[TASK] Picking up {obj_name}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        target_obj=obj_name,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    # Now place in first available drawer
                    for drawer_name, drawer_pos in positions.items():
                        if 'drawer' in drawer_name:
                            print(f"[TASK] Moving to {drawer_name} at {drawer_pos} to place {obj_name}")
                            obs, reward, done = execute_go(
                                env,
                                task,
                                source_pos=None,
                                target_pos=drawer_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                timeout=10.0
                            )
                            print(f"[TASK] Attempting to place {obj_name} into {drawer_name}")
                            try:
                                obs, reward, done = execute_place(
                                    env,
                                    task,
                                    target_obj=obj_name,
                                    target_drawer=drawer_name,
                                    target_pos=drawer_pos,
                                    approach_distance=0.15,
                                    max_steps=100,
                                    threshold=0.01,
                                    approach_axis='z',
                                    timeout=10.0
                                )
                                print(f"[TASK] Place action complete for {obj_name} in {drawer_name}")
                            except Exception as place_ex:
                                print(f"[TASK] Place failed: {place_ex}")
                            break
                    break
        except Exception as exec_ex:
            print(f"[TASK] Oracle Plan failed due to: {exec_ex}")
            # You can add more fallback logic or diagnostics here if needed

        # TODO: Insert any additional plan steps to fully achieve your task's goal.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
