# run_skeleton_task.py (Completed -- with Exploration Phase for Missing Predicate)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # To be robust, catch errors on missing keys
        try:
            # Example object and drawer naming
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            object_name = 'object1'
            robot_init_pos = positions.get('robot', positions.get('robot_init', 'ready-pose'))
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            object_pos = positions.get(object_name, None)
        except Exception as e:
            print("[Error] While retrieving positions: ", str(e))
            return

        # === Exploration Phase ===
        # Per feedback, (drawer-unlocked drawer1) is the missing predicate, so we must explore if the drawer is unlocked.
        # From exploration knowledge and domain, 'execute_pull' can help determine lock state, but only after trying to pull.
        # We'll pretend the plan is:
        #  1. Go to drawer
        #  2. Attempt to pull the drawer handle using execute_pull
        # If fail, realize (drawer-unlocked drawer1) might be missing.

        print("[Exploration] Checking if drawer is unlocked by attempting pull...")
        # We assume we have to pick up the handle before pulling
        try:
            # Move to handle location if needed (using execute_go)
            if handle_pos is not None:
                print(f"[Exploration] Moving robot to handle at {handle_pos}...")
                obs, reward, done, info = execute_go(
                    env, 
                    task, 
                    robot_init_pos, 
                    handle_pos
                )
                if done:
                    print("[Exploration] Task complete after move. (Unexpected)")
                    return
            
            # Pick the handle - (assuming handle is pickable with execute_pick)
            print(f"[Exploration] Picking up handle '{handle_name}' at {handle_pos}...")
            obs, reward, done, info = execute_pick(
                env, 
                task, 
                handle_name, 
                handle_pos
            )
            if done:
                print("[Exploration] Task complete after picking handle. (Unexpected)")
                return

            # Attempt to pull the drawer -- this should work only if drawer-unlocked is true.
            print(f"[Exploration] Attempting to pull drawer '{drawer_name}' with handle '{handle_name}' at {drawer_pos}...")

            obs, reward, done, info = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                drawer_pos
            )
            print("[Exploration] Pull attempted.")
        except Exception as e:
            print("[Exploration] Pull failed, inferring missing predicate likely (drawer-unlocked drawer1). Exception:", str(e))

        # (After exploration, we would record/extract that (drawer-unlocked drawer1) is the missing precondition for the pull.)

        # === Normal Plan Follows: Example Oracle Plan Using Predefined Skills ===
        # This section would invoke the oracle plan step by step.
        # For demonstration, here is a typical robotic drawer task (edit as needed for your oracle plan):

        # 1. Move to object (if not already near)
        if object_pos is not None:
            print(f"[Plan] Moving robot to object at {object_pos}...")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    task,
                    handle_pos if handle_pos is not None else robot_init_pos,
                    object_pos
                )
                if done:
                    print("[Plan] Task ended after moving to object!")
                    return
            except Exception as e:
                print("[Plan] Error during move to object:", str(e))
                return

        # 2. Pick up the object (if not already holding)
        try:
            print(f"[Plan] Picking up object '{object_name}' at {object_pos}...")
            obs, reward, done, info = execute_pick(
                env,
                task,
                object_name,
                object_pos
            )
            if done:
                print("[Plan] Task ended after picking object!")
                return
        except Exception as e:
            print("[Plan] Error during pick:", str(e))
            return

        # 3. Move to drawer (if not already at drawer)
        if drawer_pos is not None:
            print(f"[Plan] Moving to drawer at {drawer_pos}...")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    task,
                    object_pos,
                    drawer_pos
                )
                if done:
                    print("[Plan] Task ended after move to drawer!")
                    return
            except Exception as e:
                print("[Plan] Error during move to drawer:", str(e))
                return

        # 4. Place object into drawer
        try:
            print(f"[Plan] Placing object '{object_name}' into drawer '{drawer_name}' at {drawer_pos}...")
            obs, reward, done, info = execute_place(
                env,
                task,
                object_name,
                drawer_name,
                drawer_pos
            )
            if done:
                print("[Plan] Task ended after placing object in drawer!")
                return
        except Exception as e:
            print("[Plan] Error during place:", str(e))
            return

        # 5. Push the drawer closed
        try:
            print(f"[Plan] Pushing drawer '{drawer_name}' closed at {drawer_pos}...")
            obs, reward, done, info = execute_push(
                env,
                task,
                drawer_name,
                drawer_pos
            )
            if done:
                print("[Plan] Task ended after pushing drawer!")
                return
        except Exception as e:
            print("[Plan] Error during push:", str(e))
            return

        print("[Plan] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()