# run_skeleton_task.py (Filled and Generic, with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all available skills as implementations

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions & Names ===
        positions = get_object_positions()
        # positions: dictionary, e.g., {'drawer1': (x, y, z), ...}

        # === Exploration to Find Missing Predicate ===
        print("[Exploration] Starting exploration phase to detect missing predicate(s).")

        exploration_predicates = [
            'identified',
            'temperature-known',
            'weight-known',
            'durability-known',
            'lock-known',
        ]

        # For demonstration, assume we need to learn lock-known for objects like drawers/handles.
        objects = list(positions.keys())
        explored_predicates = {}
        missing_predicate = None

        # Try each skill and see which predicates become satisfied.
        for obj in objects:
            obj_pos = positions[obj]
            print(f"[Exploration] Testing predicates for object '{obj}' at position {obj_pos}")
            explored_predicates[obj] = {}

            # Attempt execute_pick (if available and object on floor)
            try:
                print(f"[Exploration] Trying 'execute_pick' on {obj}")
                obs, reward, done = execute_pick(env, task, target=obj, position=obj_pos)
                explored_predicates[obj]['picked'] = True
            except Exception as e:
                print(f"[Exploration] execute_pick failed on {obj}: {e}")
                explored_predicates[obj]['picked'] = False

            # Try execute_pull (to trigger 'lock-known')
            try:
                print(f"[Exploration] Trying 'execute_pull' on {obj}")
                obs, reward, done = execute_pull(env, task, target=obj, position=obj_pos)
                print(f"[Exploration] Success: 'lock-known' might be revealed for {obj}")
                explored_predicates[obj]['lock-known'] = True
                missing_predicate = 'lock-known'
                break  # Found missing predicate
            except Exception as e:
                print(f"[Exploration] execute_pull failed on {obj}: {e}")
                explored_predicates[obj]['lock-known'] = False

        if missing_predicate is not None:
            print(f"[Exploration] Identified missing predicate: {missing_predicate}")
        else:
            print("[Exploration] No missing core predicate found during exploration.")

        # (Optional) If needed, log or process other exploration acts (pick_weight, pick_durability, move/identify etc).

        # === Main Oracle Plan Execution ===
        # Placeholder: you would insert your main plan execution using the available skills.
        # E.g., move robot, pick/place, handle drawer, etc. Here we only demonstrate exploration.
        # For the actual task, you would use calls such as:
        #
        # obs, reward, done = execute_go(env, task, from_location, to_location)
        # obs, reward, done = execute_pick(env, task, target=obj, position=pos)
        # obs, reward, done = execute_pull(env, task, target=handle, position=handle_pos)
        #

        print("[Task] Exploration and predicate detection phase completed.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()